from flask import Flask, render_template, jsonify, request
from mysql import connector
import uuid
from kubernetes import client, config
import time
import os
import yaml
from kubernetes.stream import stream

app = Flask(__name__)

@app.route('/')
def index():
    return render_template('catalog.html')

# Route to fetch catalog from MySQL database
@app.route('/api/catalog')
def catalog():

    # Connection to the MySQL Database
    conn = connector.connect(
        host='mysql-service',
        user='admin',
        password='adminPass',
        database='mediadb',
    )
    cursor = conn.cursor(dictionary=True)
    cursor.execute("SELECT title, description, length FROM catalog")
    catalog_data = cursor.fetchall()
    cursor.close()
    conn.close()
    return jsonify(catalog_data)

# Route to create a visualization environment and return the URL
@app.route('/api/visualization')
def visualization():
    # Load kube config
    config.load_kube_config()

    # Unique environment name
    env_id = str(uuid.uuid4())[:8]
    namespace = f"visualizer-{env_id}"

    # Kubernetes API clients
    v1 = client.CoreV1Api()
    apps_v1 = client.AppsV1Api()

    # Create Namespace
    ns = client.V1Namespace(
        metadata=client.V1ObjectMeta(name=namespace)
    )
    try:
        v1.create_namespace(ns)
    except client.exceptions.ApiException as e:
        if e.status == 409:
            return f"Namespace {namespace} already exists.", 409
        else:
            raise e

    # Apply PersistentVolumeClaim from YAML
    with open(os.path.join('k8s', 'configmaps', 'pvc.yaml')) as f:
        pvc_data = yaml.safe_load(f)
        pvc = client.V1PersistentVolumeClaim(
            api_version=pvc_data['apiVersion'],
            kind=pvc_data['kind'],
            metadata=pvc_data['metadata'],
            spec=pvc_data['spec']
        )
        v1.create_namespaced_persistent_volume_claim(namespace=namespace, body=pvc)

    # Fetch media file from MySQL (mediadb) using id from POST request
    media_id = request.args.get('id') or request.form.get('id')
    if not media_id:
        return jsonify({"error": "Missing media id"}), 400

    # Connect to mediadb and fetch file content
    conn = connector.connect(
        host='mysql-service',
        user='admin',
        password='adminPass',
        database='mediadb',
    )
    cursor = conn.cursor(dictionary=True)
    cursor.execute("SELECT file FROM media WHERE id = %s", (media_id,))
    row = cursor.fetchone()
    cursor.close()
    conn.close()

    if not row or 'file' not in row:
        return jsonify({"error": "Media not found"}), 404

    media_file_path = f"/tmp/{media_id}"
    with open(media_file_path, "wb") as f:
        f.write(row['file'])


        # Copy the media file into the PVC by creating a temporary pod that mounts the PVC

        copy_pod_manifest = {
            "apiVersion": "v1",
            "kind": "Pod",
            "metadata": {"name": "media-copy-pod"},
            "spec": {
                "containers": [{
                    "name": "copy",
                    "image": "busybox",
                    "command": ["sleep", "3600"],
                    "volumeMounts": [{
                        "name": "media-pvc",
                        "mountPath": "/media"
                    }]
                }],
                "volumes": [{
                    "name": "media-pvc",
                    "persistentVolumeClaim": {
                        "claimName": pvc_data['metadata']['name']
                    }
                }],
                "restartPolicy": "Never"
            }
        }

        # Create the copy pod
        v1.create_namespaced_pod(namespace=namespace, body=copy_pod_manifest)

        # Wait for pod to be running
        for _ in range(30):
            pod = v1.read_namespaced_pod("media-copy-pod", namespace)
            if pod.status.phase == "Running":
                break
            time.sleep(2)
        else:
            raise RuntimeError("Copy pod did not start in time")

        # Use Kubernetes API to copy the file into the pod (requires kubectl or stream API)

        with open(media_file_path, "rb") as f:
            file_data = f.read()

        exec_command = [
            '/bin/sh',
            '-c',
            f'cat > /mnt/media/{media_id}'
        ]

        resp = stream(
            v1.connect_get_namespaced_pod_exec,
            "media-copy-pod",
            namespace,
            command=exec_command,
            stderr=True, stdin=True,
            stdout=True, tty=False,
            _preload_content=False
        )
        resp.write_stdin(file_data)
        resp.close()

        # Delete the copy pod after copying
        v1.delete_namespaced_pod("media-copy-pod", namespace)

    # Media Player Deployment and Service
    apps_v1 = client.AppsV1Api()
    with open(os.path.join('k8s', 'deployments', 'deployment.yaml')) as f:
        player_deployment_data = yaml.safe_load(f)
        player_deployment = client.V1Deployment(
            api_version=player_deployment_data['apiVersion'],
            kind=player_deployment_data['kind'],
            metadata=player_deployment_data['metadata'],
            spec=player_deployment_data['spec']
        )
        apps_v1.create_namespaced_deployment(namespace=namespace, body=player_deployment)

    # Apply Media Player Service from YAML
    with open(os.path.join('k8s', 'services', 'service.yaml')) as f:
        service_data = yaml.safe_load(f)
        service = client.V1Service(
            api_version=service_data['apiVersion'],
            kind=service_data['kind'],
            metadata=service_data['metadata'],
            spec=service_data['spec']
        )
        v1.create_namespaced_service(namespace=namespace, body=service)

    # Wait for LoadBalancer IP/hostname
    lb_url = None
    for _ in range(30):
        svc = v1.read_namespaced_service(namespace, namespace)
        ingress = svc.status.load_balancer.ingress
        if ingress:
            lb_url = ingress[0].hostname if ingress[0].hostname else ingress[0].ip
            break
        time.sleep(2)

    if lb_url:
        frontend_url = f"http://{lb_url}"
        return jsonify({"url": frontend_url})
    else:
        return jsonify({"error": "LoadBalancer not ready"}), 503

if __name__ == '__main__':
    app.run(debug=True)
