import os
import sys

from PyQt6.QtWidgets import QGroupBox, QVBoxLayout, QFormLayout, QLineEdit, QPushButton, QHBoxLayout
from PyQt6.QtGui import QIcon

if getattr(sys, 'frozen', False): # PyInstaller adds this attribute
    ASSETS_DIR = os.path.join(sys._MEIPASS, "assets") 
else: 
    ASSETS_DIR = os.path.dirname(os.path.abspath(__file__)) + "/../assets"


class WifiGroup(QGroupBox):
    """
    WifiGroup is a class that represents a group of widgets for configuring the Wi-Fi settings.
    """
    
    def __init__(self, title: str) -> None:
        super().__init__(title)
        # << -- Wifi Group Setup -- >>
        self._layout = QFormLayout()
        
        # << -- Wifi SSID Setup -- >>
        self._ssid_edit = QLineEdit()
        self._ssid_edit.setPlaceholderText("My-WiFi")
        
        # << -- Wifi Password Setup -- >>
        self._password_edit = QLineEdit()
        self._password_edit.setEchoMode(QLineEdit.EchoMode.Password)
        self._password_button = QPushButton()
        self._password_button.clicked.connect(self._switch_password_echo_mode)
        self._password_button.setCheckable(True)
        self._password_button.setMaximumSize(30, 30)
        self._password_button.setIcon(QIcon(f"{ASSETS_DIR}/show.png"))
        self._password_layout = QHBoxLayout()
        self._password_layout.addWidget(self._password_edit)
        self._password_layout.addWidget(self._password_button)
    
        # << -- Wifi Form Setup -- >>
        self._layout.addRow("SSID", self._ssid_edit)
        self._layout.addRow("Password", self._password_layout)
        
        # << -- Wifi Layout Setup -- >>
        self.setLayout(self._layout)


    def _switch_password_echo_mode(self):
        """Switches the echo mode of the password edit field between normal and password."""
        if self._password_button.isChecked():
            self._password_edit.setEchoMode(QLineEdit.EchoMode.Normal)
            self._password_button.setIcon(QIcon(f"{ASSETS_DIR}/hide.png"))
        else:
            self._password_edit.setEchoMode(QLineEdit.EchoMode.Password)
            self._password_button.setIcon(QIcon(f"{ASSETS_DIR}/show.png"))

    @property
    def wifi_ssid(self) -> str:
        return self._ssid_edit.text()
    
    @wifi_ssid.setter
    def wifi_ssid(self, ssid: str) -> None:
        self._ssid_edit.setText(ssid)

    
    @property
    def wifi_password(self) -> str:
        return self._password_edit.text()

    

