from PyQt6.QtWidgets import QWidget, QVBoxLayout, QComboBox, QLineEdit, QPushButton, QGroupBox, QListWidget, QLabel, QSpacerItem, QSizePolicy


class ControlPanel(QWidget):

    def __init__(self, cameras_id: list[str]):
        super().__init__()

        # << -- Camera Section Setup -- >>
        self._camera_groupbox = QGroupBox("Camera Selector")
        self._camera_groupbox.setFixedHeight(75)
        self.camera_combobox = QComboBox()
        self.camera_combobox.setPlaceholderText("Select Camera")
        for camera_id in cameras_id:
            self.camera_combobox.addItem("ESP32CAM-" + camera_id, camera_id)
        self.camera_combobox.setCurrentIndex(0)
        self._camera_layout = QVBoxLayout()
        self._camera_layout.addWidget(self.camera_combobox)
        self._camera_groupbox.setLayout(self._camera_layout)

        # << -- Spacer Setup -- >>
        self._spacer = QSpacerItem(0, 0, QSizePolicy.Policy.Minimum, QSizePolicy.Policy.Expanding)

        # << -- Telegram Notifications Section Setup -- >>
        self._telegram_groupbox = QGroupBox("Telegram Notifications")
        self.telegram_chat_id = QLineEdit()
        self.telegram_chat_id.setPlaceholderText("Telegram Chat ID")
        self.add_notification_button = QPushButton("Add Notification")
        self._notification_list_title = QLabel("Chat IDs to Notify")
        self._notification_list = QListWidget()
        self.remove_notification_button = QPushButton("Remove Notification")
        self.remove_notification_button.setEnabled(False)
        self._notification_list.itemClicked.connect(self._enable_remove_notification_button)
        self._telegram_layout = QVBoxLayout()
        self._telegram_layout.addWidget(self.telegram_chat_id)
        self._telegram_layout.addWidget(self.add_notification_button)
        self._telegram_layout.addWidget(self._notification_list_title)
        self._telegram_layout.addWidget(self._notification_list)
        self._telegram_layout.addWidget(self.remove_notification_button)
        self._telegram_groupbox.setLayout(self._telegram_layout)
        self._telegram_groupbox.setHidden(True)


        # << -- Layout Setup -- >>
        self._layout = QVBoxLayout()
        self._layout.addWidget(self._camera_groupbox)
        self._layout.addItem(self._spacer)
        self._layout.addWidget(self._telegram_groupbox)
        self.setLayout(self._layout)


    def _enable_remove_notification_button(self) -> None:
        self.remove_notification_button.setEnabled(True)
        self.remove_notification_button.setStyleSheet("background-color: red;")


    def set_notification_list(self, client_id_list: list[str]) -> None:
        self._notification_list.clear()
        self._notification_list.addItems(client_id_list)


    def get_selected_camera_id(self) -> str:
        """
        Returns the currently selected camera ID.

        Returns:
            str: The selected camera ID.
        """
        return self.camera_combobox.currentData()
    

    def get_chat_id(self) -> str:
        """
        Returns the chat ID entered in the text box.

        Returns:
            str: The chat ID.
        """
        return self.telegram_chat_id.text()
    

    def get_selected_chat_id(self) -> str:
        """
        Returns the chat ID entered in the text box.

        Returns:
            str: The chat ID.
        """
        return self._notification_list.currentItem().text() if self._notification_list.currentItem() else None
    

    def show_telegram_notifications(self) -> None:
        """
        Shows the telegram notifications section.
        """
        self._telegram_groupbox.setHidden(False)
        self._spacer.changeSize(0, 0, QSizePolicy.Policy.Minimum, QSizePolicy.Policy.Fixed)
        self._layout.update()