from PyQt6.QtWidgets import QWidget, QVBoxLayout, QHBoxLayout, QGraphicsScene, QDialog, QLabel

from PyQt6.QtCore import Qt

from View.CaptureViewer import CaptureViewer
from View.ControlPanel import ControlPanel

class ControlView(QDialog):

    def __init__(self, cameras: dict[str, QGraphicsScene], parent=None, dark_mode=False) -> None:
        super().__init__(parent)
        self._cameras = cameras
        self.notifiers: dict[str, list] = {}

        # << -- Window Setup -- >>
        self.setWindowTitle("Control Panel")
        if dark_mode:
            self.setStyleSheet("background-color: #222222; color: #FFFFFF;")
        else:
            self.setStyleSheet("background-color: #ffffff; color: #000000;")
        self.setMinimumSize(400, 300)
        self.setModal(True)

        # << -- Control Panel Setup -- >>
        self.control_panel = ControlPanel(cameras.keys())
        self.control_panel.camera_combobox.currentIndexChanged.connect(self.update_view)
        
        # << -- Capture Viewer Setup -- >>
        self._right_layout = QVBoxLayout()  
        self._camera_id_label = QLabel()
        first_camera_id = self.control_panel.get_selected_camera_id()
        self.capture_viewer = CaptureViewer(self._cameras[first_camera_id])
        self._camera_id_label.setText(f"ESP32CAM-{first_camera_id}")
        self._camera_id_label.setStyleSheet("font-size: 18px;")
        self._camera_id_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self._right_layout.addWidget(self._camera_id_label)
        self._right_layout.addWidget(self.capture_viewer)
        self._right_layout.setAlignment(Qt.AlignmentFlag.AlignCenter)

        # << -- Layout Setup -- >>
        self._layout = QHBoxLayout()
        self._layout.addWidget(self.control_panel, 1)
        self._layout.addLayout(self._right_layout, 4)
        self.setLayout(self._layout)


    def update_view(self) -> None:
        """
        Updates the capture viewer with the selected camera.
        """
        camera_id = self.control_panel.get_selected_camera_id()
        scene = self._cameras[camera_id]
        self.capture_viewer.set_scene(scene)
        self._camera_id_label.setText(f"ESP32CAM-{camera_id}")

        if camera_id in self.notifiers:
            self.control_panel.set_notification_list(self.notifiers[camera_id])
        else:
            self.control_panel.set_notification_list([])
            self.control_panel.remove_notification_button.setEnabled(False)
            # Button Background color to default
            self.control_panel.remove_notification_button.setStyleSheet("") 

        
    def get_selected_camera_id(self) -> str:
        """
        Returns the currently selected camera ID.
        """
        return self.control_panel.get_selected_camera_id()
    

    def get_current_scene(self) -> QGraphicsScene:
        """
        Returns the current scene of the capture viewer.
        """
        return self.capture_viewer.get_scene()
    
    def add_chat_id(self, camera_id: str, chat_id: str) -> None:
        """
        Adds a chat ID to the list of notifiers for the given camera ID.
        """
        if camera_id in self.notifiers:
            self.notifiers[camera_id].append(chat_id)
        else:
            self.notifiers[camera_id] = [chat_id]

    def remove_chat_id(self, camera_id: str, chat_id: str) -> None:
        """
        Removes a chat ID from the list of notifiers for the given camera ID.
        """
        if camera_id in self.notifiers:
            self.notifiers[camera_id].remove(chat_id)
            if len(self.notifiers[camera_id]) == 0:
                del self.notifiers[camera_id]
