from pydantic import BaseModel, Field
from typing import List, Dict, Any, Optional
from uuid import UUID
from enum import Enum

from app.model.base import ServicioBase, ServicioBaseCreate, ServicioBaseUpdate, ServicioBaseOut, Address


class TipoEntrega(str, Enum):
    SERVICIO_DOMICILIO = "SERVICIO_DOMICILIO"
    RECOGIDA_LOCAL = "RECOGIDA_LOCAL"


class TipoCatering(str, Enum):
    BANQUETE_IMPERIAL = "BANQUETE_IMPERIAL"
    BUFET = "BUFET"
    COCKTAIL = "COCKTAIL"
    FAMILIAR = "FAMILIAR"


class Alergeno(str, Enum):
    GLUTEN = "GLUTEN"
    LACTEOS = "LACTEOS"
    FRUTOS_DE_CASCARA = "FRUTOS_DE_CASCARA"
    HUEVOS = "HUEVOS"
    PESCADO = "PESCADO"
    CRUSTACEOS = "CRUSTACEOS"
    MOLUSCOS = "MOLUSCOS"
    SOJA = "SOJA"
    APIO = "APIO"
    MOSTAZA = "MOSTAZA"
    SESAMO = "SESAMO"
    ALTRAMUCES = "ALTRAMUCES"
    SULFITOS = "SULFITOS"


class TipoMenu(str, Enum):
    MEDITERRANEO = "MEDITERRANEO"
    CONTINENTAL = "CONTINENTAL"
    ASIATICO = "ASIATICO"
    VEGETARIANO = "VEGETARIANO"
    VEGANO = "VEGANO"


class MenuCreate(BaseModel):
    type_menu: TipoMenu
    nombre: str
    descripcion: str
    platos: List[str]
    opciones_especiales: List[str]


class MenuUpdate(BaseModel):
    type_menu: Optional[TipoMenu] = None
    nombre: Optional[str] = None
    descripcion: Optional[str] = None
    platos: Optional[List[str]] = None
    opciones_especiales: Optional[List[str]] = None


class Menu(BaseModel):
    type_menu: TipoMenu
    nombre: str
    descripcion: str
    platos: List[str]
    opciones_especiales: List[str]


class MenuOut(Menu):
    class Config:
        from_attributes = True


class CateringDetailsCreate(BaseModel):
    tipo_entrega: TipoEntrega
    tipo_catering: TipoCatering
    min_comensales: int
    max_comensales: int
    duracion: float
    alergenos: List[Alergeno]
    caracteristicas: List[str]
    menu: MenuCreate


class CateringDetailsUpdate(BaseModel):
    tipo_entrega: Optional[TipoEntrega] = None
    tipo_catering: Optional[TipoCatering] = None
    min_comensales: Optional[int] = None
    max_comensales: Optional[int] = None
    duracion: Optional[float] = None
    alergenos: Optional[List[Alergeno]] = None
    caracteristicas: Optional[List[str]] = None
    menu: Optional[MenuUpdate] = None


class CateringDetails(BaseModel):
    id: UUID
    tipo_entrega: TipoEntrega
    tipo_catering: TipoCatering
    min_comensales: int
    max_comensales: int
    duracion: float
    alergenos: List[Alergeno]
    caracteristicas: List[str]
    menu: Menu


class CateringDetailsOut(CateringDetails):
    class Config:
        from_attributes = True


class CateringCreate(ServicioBaseCreate):
    catering: CateringDetailsCreate


class CateringUpdate(ServicioBaseUpdate):
    catering: Optional[CateringDetailsUpdate] = None


class Catering(ServicioBase):
    catering: CateringDetails


class CateringOut(ServicioBaseOut):
    catering: CateringDetailsOut
    
    class Config:
        from_attributes = True 