"""
Espai service routes
"""
from typing import List, Optional, Dict
from uuid import UUID
from fastapi import APIRouter, HTTPException, Query
from sqlmodel import Session

from app.models import ServiceEspaiOut, ServiceCreateEspai, ServiceUpdateEspai, ServiceEspaisOut, Service, Espai, EspaiType
from app import crud
from app.api.deps import (
    CurrentUser,
    SessionDep,
    get_current_active_superuser,
)
router = APIRouter()


@router.post("/", response_model=ServiceEspaiOut)
def create_espai(
    espai_in: ServiceCreateEspai,
    session: SessionDep,
    current_user: CurrentUser
) -> ServiceEspaiOut:
    if not current_user.is_superuser and not crud.user.is_user_vendor(session=session, user_id=current_user.id):
        raise HTTPException(status_code=403, detail="User must be a vendor or superuser")

    return crud.espai.create_espai_service(session=session, obj_in=espai_in, account_id=current_user.id)



@router.put("/{espai_id}", response_model=ServiceEspaiOut)
def update_espai(
    espai_id: UUID,
    espai_in: ServiceUpdateEspai,
    session: SessionDep,
    current_user: CurrentUser
) -> ServiceEspaiOut:
    service = session.get(Service, espai_id)
    espai = session.get(Espai, espai_id)
    if service.account_id != current_user.id and not current_user.is_superuser:
        raise HTTPException(status_code=403, detail="Not enough permissions")
    return crud.espai.update_espai_service(session=session, service=service, espai=espai, obj_in=espai_in)


@router.patch("/{espai_id}", response_model=ServiceEspaiOut)
def patch_espai(
    espai_id: UUID,
    espai_in: ServiceUpdateEspai,
    session: SessionDep,
    current_user: CurrentUser
) -> ServiceEspaiOut:
    service = session.get(Service, espai_id)
    espai = session.get(Espai, espai_id)
    if service.account_id != current_user.id and not current_user.is_superuser:
        raise HTTPException(status_code=403, detail="Not enough permissions")
    return crud.espai.update_espai_service(session=session, service=service, espai=espai, obj_in=espai_in)


@router.delete("/{espai_id}")
def delete_espai(
    espai_id: UUID,
    session: SessionDep,
    current_user: CurrentUser
):
    service = session.get(Service, espai_id)
    espai = session.get(Espai, espai_id)
    if service.account_id != current_user.id and not current_user.is_superuser:
        raise HTTPException(status_code=403, detail="Not enough permissions")
    crud.espai.delete_espai_service(session=session, service=service, espai=espai)
    return {"Servicio de espai eliminado correctamente": True}


@router.get("/by_id/{espai_id}", response_model=ServiceEspaiOut)
def get_espai_by_id(
    espai_id: UUID,
    session: SessionDep,
) -> ServiceEspaiOut:
    return crud.espai.get_espai_service(session=session, service_id=espai_id)


@router.get("/all", response_model=ServiceEspaisOut)
def get_all_espais(
    session: SessionDep,
    skip: int = Query(0, ge=0),
    limit: int = Query(10, ge=1, le=100)
) -> ServiceEspaisOut:
    return crud.espai.get_all_espais(db=session, skip=skip, limit=limit)


@router.get("/all_me", response_model=ServiceEspaisOut)
def get_all_espais_me(
    session: SessionDep,
    current_user: CurrentUser,
    skip: int = Query(0, ge=0),
    limit: int = Query(10, ge=1, le=100)
) -> ServiceEspaisOut:
    return crud.espai.get_all_espais_me(db=session, account_id=current_user.id, skip=skip, limit=limit)


@router.get("/filter", response_model=ServiceEspaisOut)
def get_espais_filtered(
    session: SessionDep,
    espai_type: Optional[EspaiType] = None,
    num_capacitat: Optional[int] = None,
    caracteristicas: Optional[List[str]] = Query(None),
    skip: int = Query(0, ge=0),
    limit: int = Query(10, ge=1, le=100)
) -> ServiceEspaisOut:
    return crud.espai.get_espais_filtered(
        db=session,
        espai_type=espai_type,
        num_capacitat=num_capacitat,
        caracteristicas=caracteristicas,
        skip=skip,
        limit=limit
    )


@router.get("/filter_me", response_model=ServiceEspaisOut)
def get_espais_filtered_me(
    session: SessionDep,
    current_user: CurrentUser,
    espai_type: Optional[EspaiType] = None,
    num_capacitat: Optional[int] = None,
    caracteristicas: Optional[List[str]] = Query(None),
    skip: int = Query(0, ge=0),
    limit: int = Query(10, ge=1, le=100)
) -> ServiceEspaisOut:
    return crud.espai.get_espais_filtered(
        db=session,
        espai_type=espai_type,
        num_capacitat=num_capacitat,
        caracteristicas=caracteristicas,
        account_id=current_user.id,
        skip=skip,
        limit=limit
    )