""" Espai Propi endpoints """
from typing import List, Optional
from uuid import UUID
from fastapi import APIRouter, Depends, HTTPException
from sqlmodel import Session, select

from app.api.deps import (
    CurrentUser,
    SessionDep,
    get_current_active_superuser,
)
from app import crud
from app.models import (
    EspaiPropiCreate,
    EspaiPropiUpdate,
    EspaiPropiOut,
    EspaiPropisOut,
    TipusEspaiPropi,
    EspaiPropi
)

router = APIRouter()

@router.post("/espai-propi", response_model=EspaiPropiOut)
def create_espai_propi(
    *,
    espai_propi_in: EspaiPropiCreate,
    session: SessionDep,
    current_user: CurrentUser
) -> EspaiPropiOut:
    """
    Create a new espai propi.
    """
    return crud.espai_propi.create_espai_propi(
        session=session,
        espai_propi_in=espai_propi_in,
        user=current_user
    )

@router.put("/espai-propi/{espai_propi_id}", response_model=EspaiPropiOut)
def update_espai_propi(
    *,
    espai_propi_id: UUID,
    espai_propi_in: EspaiPropiUpdate,
    session: SessionDep,
    current_user: CurrentUser
) -> EspaiPropiOut:
    """
    Update an espai propi.
    """
    # Get the espai propi
    espai_propi = session.get(EspaiPropi, espai_propi_id)
    if not espai_propi:
        raise HTTPException(status_code=404, detail="Espai propi not found")
    
    # Check if the espai propi belongs to the current user
    if espai_propi.account_id != current_user.id:
        raise HTTPException(status_code=403, detail="Not enough permissions")
    
    return crud.espai_propi.update_espai_propi(
        session=session,
        espai_propi=espai_propi,
        espai_propi_in=espai_propi_in
    )

@router.get("/espai-propi/{espai_propi_id}", response_model=EspaiPropiOut)
def get_espai_propi(
    *,
    espai_propi_id: UUID,
    session: SessionDep,
    current_user: CurrentUser
) -> EspaiPropiOut:
    """
    Get espai propi by ID.
    """
    return crud.espai_propi.get_espai_propi_by_id(
        session=session,
        id=espai_propi_id
    )

@router.get("/espai-propi", response_model=EspaiPropisOut)
def get_espai_propis(
    *,
    skip: int = 0,
    limit: int = 100,
    session: SessionDep,
    current_user: CurrentUser
) -> EspaiPropisOut:
    """
    Get all espais propis.
    """
    if not current_user.is_superuser:
        raise HTTPException(status_code=403, detail="Not enough permissions")
    return crud.espai_propi.get_espai_propis(
        session=session,
        skip=skip,
        limit=limit,
    )

@router.get("/my-espai-propi", response_model=EspaiPropisOut)
def get_my_espai_propis(
    *,
    skip: int = 0,
    limit: int = 100,
    session: SessionDep,
    current_user: CurrentUser
) -> EspaiPropisOut:
    """
    Get all espais propis for the current user.
    """
    return crud.espai_propi.get_espai_propis_by_account(
        session=session,
        account_id=current_user.id,
        skip=skip,
        limit=limit
    )

@router.delete("/espai-propi/{espai_propi_id}")
def delete_espai_propi(
    *,
    espai_propi_id: UUID,
    session: SessionDep,
    current_user: CurrentUser
) -> None:
    """
    Delete an espai propi.
    """
    # Get the espai propi usando select para evitar la validación del enum
    statement = select(EspaiPropi).where(EspaiPropi.id == espai_propi_id)
    espai_propi = session.exec(statement).first()
    if not espai_propi:
        raise HTTPException(status_code=404, detail="Espai propi not found")
    
    # Check if the espai propi belongs to the current user
    if espai_propi.account_id != current_user.id:
        raise HTTPException(status_code=403, detail="Not enough permissions")
    
    crud.espai_propi.delete_espai_propi(
        session=session,
        espai_propi=espai_propi
    )
    return {"message": "Espai propi deleted successfully"}


@router.patch("/espai-propi/{espai_propi_id}", response_model=EspaiPropiOut)
def patch_espai_propi(
    *,
    espai_propi_id: UUID,
    espai_propi_in: EspaiPropiUpdate,
    session: SessionDep,
    current_user: CurrentUser
) -> EspaiPropiOut:
    """
    Update an espai propi partially.
    """
    # Get the espai propi usando select para evitar la validación del enum
    statement = select(EspaiPropi).where(EspaiPropi.id == espai_propi_id)
    espai_propi = session.exec(statement).first()
    if not espai_propi:
        raise HTTPException(status_code=404, detail="Espai propi not found")
    
    # Check if the espai propi belongs to the current user
    if espai_propi.account_id != current_user.id:
        raise HTTPException(status_code=403, detail="Not enough permissions")
    
    return crud.espai_propi.update_espai_propi(
        session=session,
        espai_propi=espai_propi,
        espai_propi_in=espai_propi_in
    )