""" Fotografía service routes """
from typing import List, Optional, Any
from uuid import UUID

from fastapi import APIRouter, HTTPException, Query
from sqlmodel import Session

from app.api.deps import CurrentUser, SessionDep, get_current_active_superuser
from app import crud
from app.models import FotografiaType, ServicioEspecial, ServiceFotografiaOut, ServiceFotografiasOut, ServiceCreateFotografia, ServiceUpdateFotografia


router = APIRouter()

# -- Endpoints para obtener servicios de fotografía --

@router.get("/by_id/{fotografia_id}", response_model=ServiceFotografiaOut)
def get_fotografia(
    fotografia_id: UUID,
    session: SessionDep,
) -> ServiceFotografiaOut:
    """
    Obtener un servicio de fotografía por su ID
    """
    fotografia = crud.fotografia.get_fotografia(db=session, fotografia_id=fotografia_id)
    if not fotografia:
        raise HTTPException(status_code=404, detail="Servicio de fotografía no encontrado")
    
    return fotografia

@router.get("/all", response_model=ServiceFotografiasOut)
def get_fotografias(
    session: SessionDep,
    skip: int = Query(0, ge=0),
    limit: int = Query(10, ge=1, le=100)
) -> ServiceFotografiasOut:
    """
    Obtener todos los servicios de fotografía (paginados)
    """
    # Administradores pueden ver todos los servicios
    fotografias, total = crud.fotografia.get_fotografias(db=session, skip=skip, limit=limit)
    return ServiceFotografiasOut(data=fotografias, count=total)

@router.get("/all_me", response_model=ServiceFotografiasOut)
def get_fotografias_me(
    session: SessionDep,
    current_user: CurrentUser,
    skip: int = Query(0, ge=0),
    limit: int = Query(10, ge=1, le=100)
) -> ServiceFotografiasOut:
    """
    Obtener todos los servicios de fotografía (paginados)
    """
    # Usuarios normales solo ven sus propios servicios
    fotografias, total = crud.fotografia.get_fotografias_by_account(
        db=session, 
        account_id=current_user.id, 
        skip=skip, 
        limit=limit
    )
    
    return ServiceFotografiasOut(data=fotografias, count=total)

@router.post("/", response_model=ServiceFotografiaOut)
def create_fotografia(
    fotografia_in: ServiceCreateFotografia,
    session: SessionDep,
    current_user: CurrentUser
) -> ServiceFotografiaOut:
    """
    Crear un nuevo servicio de fotografía
    """
    if not current_user.is_superuser and not crud.user.is_user_vendor(session=session, user_id=current_user.id):
        raise HTTPException(status_code=403, detail="User must be a vendor or superuser")

    return crud.fotografia.create_fotografia(
        db=session, 
        service_fotografia_in=fotografia_in, 
        account_id=current_user.id
    )

@router.get("/filter", response_model=ServiceFotografiasOut)
def get_fotografias_filtered(
    session: SessionDep,
    fotografia_type: Optional[FotografiaType] = None,
    servicio_especial: Optional[ServicioEspecial] = None,
    min_duracion: Optional[float] = None,
    max_duracion: Optional[float] = None,
    skip: int = Query(0, ge=0),
    limit: int = Query(10, ge=1, le=100)
) -> ServiceFotografiasOut:
    """
    Obtener servicios de fotografía con filtros opcionales:
    - Tipo de fotografía
    - Servicio especial
    - Duración mínima
    - Duración máxima
    """
    # Administradores pueden ver todos los servicios
    fotografias, total = crud.fotografia.get_fotografias_filtered(
        db=session,
        fotografia_type=fotografia_type,
        servicio_especial=servicio_especial,
        min_duracion=min_duracion,
        max_duracion=max_duracion,
        skip=skip,
        limit=limit
    )
    
    return ServiceFotografiasOut(data=fotografias, count=total)


@router.get("/filter_me", response_model=ServiceFotografiasOut)
def get_fotografias_filtered_me(
    session: SessionDep,
    current_user: CurrentUser,
    fotografia_type: Optional[FotografiaType] = None,
    servicio_especial: Optional[ServicioEspecial] = None,
    min_duracion: Optional[float] = None,
    max_duracion: Optional[float] = None,
    skip: int = Query(0, ge=0),
    limit: int = Query(10, ge=1, le=100)
) -> ServiceFotografiasOut:
    """
    Obtener servicios de fotografía con filtros opcionales:
    - Tipo de fotografía
    - Servicio especial
    - Duración mínima (en horas, puede ser decimal)
    - Duración máxima (en horas, puede ser decimal)
    """
    fotografias, total = crud.fotografia.get_fotografias_filtered(
        db=session,
        fotografia_type=fotografia_type,
        servicio_especial=servicio_especial,
        min_duracion=min_duracion,
        max_duracion=max_duracion,
        account_id=current_user.id,
        skip=skip,
        limit=limit
    )
    
    return ServiceFotografiasOut(data=fotografias, count=total)



@router.put("/{fotografia_id}", response_model=ServiceFotografiaOut)
def update_fotografia(
    fotografia_id: UUID,
    fotografia_in: ServiceUpdateFotografia,
    session: SessionDep,
    current_user: CurrentUser
) -> ServiceFotografiaOut:
    """
    Actualizar un servicio de fotografía existente
    """
    fotografia = crud.fotografia.get_fotografia(db=session, fotografia_id=fotografia_id)
    if not fotografia:
        raise HTTPException(status_code=404, detail="Servicio de fotografía no encontrado")
    
    # Verificar permisos (superusuario o propietario)
    if not current_user.is_superuser and fotografia.account_id != current_user.id:
        raise HTTPException(status_code=403, detail="No tienes permiso para modificar este servicio")
    
    return crud.fotografia.update_fotografia(
        db=session, 
        fotografia_id=fotografia_id, 
        service_fotografia_in=fotografia_in
    )

@router.patch("/{fotografia_id}", response_model=ServiceFotografiaOut)
def patch_fotografia(
    fotografia_id: UUID,
    fotografia_in: ServiceUpdateFotografia,
    session: SessionDep,
    current_user: CurrentUser
) -> ServiceFotografiaOut:
    """
    Actualizar parcialmente un servicio de fotografía existente
    """
    fotografia = crud.fotografia.get_fotografia(db=session, fotografia_id=fotografia_id)
    if not fotografia:
        raise HTTPException(status_code=404, detail="Servicio de fotografía no encontrado")
    
    # Verificar permisos (superusuario o propietario)
    if not current_user.is_superuser and fotografia.account_id != current_user.id:
        raise HTTPException(status_code=403, detail="No tienes permiso para modificar este servicio")
    
    return crud.fotografia.update_fotografia(
        db=session, 
        fotografia_id=fotografia_id, 
        service_fotografia_in=fotografia_in
    )

@router.delete("/{fotografia_id}")
def delete_fotografia(
    fotografia_id: UUID,
    session: SessionDep,
    current_user: CurrentUser
):
    """
    Eliminar un servicio de fotografía
    """
    fotografia = crud.fotografia.get_fotografia(db=session, fotografia_id=fotografia_id)
    if not fotografia:
        raise HTTPException(status_code=404, detail="Servicio de fotografía no encontrado")
    
    # Verificar permisos (superusuario o propietario)
    if not current_user.is_superuser and fotografia.account_id != current_user.id:
        raise HTTPException(status_code=403, detail="No tienes permiso para eliminar este servicio")
    
    if not crud.fotografia.delete_fotografia(db=session, fotografia_id=fotografia_id):
        raise HTTPException(status_code=404, detail="Servicio de fotografía no encontrado")
    
    return {"Servicio de fotografía eliminado correctamente"}
