""" Mobiliari i Decoracio endpoints """
from typing import List, Optional
from uuid import UUID

from fastapi import APIRouter, Depends, HTTPException, Query
from sqlmodel import Session

from app.api.deps import CurrentUser, SessionDep, get_current_active_superuser
from app import crud
from app.models import (
    MobiliariDecoracio,
    MobiliariDecoracioType,
    TipoMobiliriDecoracio,
    MobiliariDecoracioOut,
    MobiliariDecoraciosOut,
    ServiceUpdateMobiliariDecoracio,
    ServiceCreateMobiliariDecoracio,
    ServiceMobiliariDecoracioOut,
    ServiceMobiliariDecoraciosOut,
    Service
)

router = APIRouter()


@router.post("/", response_model=ServiceMobiliariDecoracioOut)
def create_mobiliari_decoracio(
    *,
    session: SessionDep,
    current_user: CurrentUser,
    mobiliari_decoracio_in: ServiceCreateMobiliariDecoracio
):
    """Create a new mobiliari decoracio service"""
    if not current_user.is_superuser and not crud.user.is_user_vendor(session=session, user_id=current_user.id):
        raise HTTPException(status_code=403, detail="User must be a vendor or superuser")

    return crud.mobiliari_decoracio.create_mobiliari_decoracio(
        session=session,
        mobiliari_decoracio_in=mobiliari_decoracio_in,
        user=current_user
    )


@router.put("/{mobiliari_decoracio_id}", response_model=ServiceMobiliariDecoracioOut)
def update_mobiliari_decoracio(
    *,
    session: SessionDep,
    current_user: CurrentUser,
    mobiliari_decoracio_id: UUID,
    mobiliari_decoracio_in: ServiceUpdateMobiliariDecoracio
):
    """Update a mobiliari decoracio service"""
    service = session.get(Service, mobiliari_decoracio_id)
    mobiliari = session.get(MobiliariDecoracio, mobiliari_decoracio_id)
    if service.account_id != current_user.id and not current_user.is_superuser:
        raise HTTPException(status_code=403, detail="Not enough permissions")
    return crud.mobiliari_decoracio.update_mobiliari_decoracio(
        session=session,
        service=service,
        mobiliari=mobiliari,
        mobiliari_decoracio_in=mobiliari_decoracio_in
    )


@router.patch("/{mobiliari_decoracio_id}", response_model=ServiceMobiliariDecoracioOut)
def update_mobiliari_decoracio_partial(
    *,
    session: SessionDep,
    current_user: CurrentUser,
    mobiliari_decoracio_id: UUID,
    mobiliari_decoracio_in: ServiceUpdateMobiliariDecoracio
):
    """Update a mobiliari decoracio service partially"""
    service = session.get(Service, mobiliari_decoracio_id)
    mobiliari = session.get(MobiliariDecoracio, mobiliari_decoracio_id)
    if service.account_id != current_user.id and not current_user.is_superuser:
        raise HTTPException(status_code=403, detail="Not enough permissions")
    return crud.mobiliari_decoracio.update_mobiliari_decoracio(
        session=session,
        service=service,
        mobiliari=mobiliari,
        mobiliari_decoracio_in=mobiliari_decoracio_in
    )


@router.delete("/{mobiliari_decoracio_id}")
def delete_mobiliari_decoracio(
    *,
    session: SessionDep,
    current_user: CurrentUser,
    mobiliari_decoracio_id: UUID
):
    """Delete a mobiliari decoracio service"""
    service = session.get(Service, mobiliari_decoracio_id)
    mobiliari = session.get(MobiliariDecoracio, mobiliari_decoracio_id)
    if service.account_id != current_user.id and not current_user.is_superuser:
        raise HTTPException(status_code=403, detail="Not enough permissions")
    crud.mobiliari_decoracio.delete_mobiliari_decoracio(
        session=session,
        service=service,
        mobiliari=mobiliari
    )
    return {"message": "Mobiliari decoracio deleted successfully"}


@router.get("/by_id/{mobiliari_decoracio_id}", response_model=ServiceMobiliariDecoracioOut)
def get_mobiliari_decoracio_by_id(
    *,
    session: SessionDep,
    mobiliari_decoracio_id: UUID
):
    """Get a mobiliari decoracio service by ID"""
    return crud.mobiliari_decoracio.get_mobiliari_decoracio_by_id(
        session=session,
        id=mobiliari_decoracio_id
    )


@router.get("/all", response_model=ServiceMobiliariDecoraciosOut)
def get_all_mobiliari_decoracio(
    *,
    session: SessionDep,
    skip: int = Query(0, ge=0),
    limit: int = Query(10, ge=1, le=100)
):
    """Get all mobiliari decoracio services"""
    return crud.mobiliari_decoracio.get_all_mobiliaris(
        session=session,
        skip=skip,
        limit=limit
    )


@router.get("/all_me", response_model=ServiceMobiliariDecoraciosOut)
def get_all_mobiliari_decoracio_me(
    *,
    session: SessionDep,
    current_user: CurrentUser,
    skip: int = Query(0, ge=0),
    limit: int = Query(10, ge=1, le=100)
):
    """Get all mobiliari decoracio services for the current user"""
    return crud.mobiliari_decoracio.get_mobiliaris_by_account(
        session=session,
        account_id=current_user.id,
        skip=skip,
        limit=limit
    )


@router.get("/filter", response_model=ServiceMobiliariDecoraciosOut)
def get_mobiliari_decoracio_filtered(
    *,
    session: SessionDep,
    service_type: Optional[MobiliariDecoracioType] = None,
    tipologia: Optional[List[TipoMobiliriDecoracio]] = Query(None),
    is_rental: Optional[bool] = None,
    cantidad: Optional[int] = None,
    material: Optional[List[str]] = Query(None, description="Lista de materiales a filtrar"),
    color: Optional[List[str]] = Query(None, description="Lista de colores a filtrar"),
    caracteristicas: Optional[List[str]] = Query(None, description="Lista de características a filtrar"),
    forma: Optional[List[str]] = Query(None, description="Lista de formas a filtrar"),
    skip: int = Query(0, ge=0),
    limit: int = Query(10, ge=1, le=100)
):
    """Get filtered mobiliari decoracio services"""
    return crud.mobiliari_decoracio.get_mobiliari_decoracios(
        session=session,
        skip=skip,
        limit=limit,
        service_type=service_type,
        tipologia=tipologia,
        is_rental=is_rental,
        cantidad=cantidad,
        material=material,
        color=color,
        caracteristicas=caracteristicas,
        forma=forma
    )


@router.get("/filter_me", response_model=ServiceMobiliariDecoraciosOut)
def get_mobiliari_decoracio_filtered_me(
    *,
    session: SessionDep,
    current_user: CurrentUser,
    service_type: Optional[MobiliariDecoracioType] = None,
    tipologia: Optional[List[TipoMobiliriDecoracio]] = Query(None),
    is_rental: Optional[bool] = None,
    cantidad: Optional[int] = None,
    material: Optional[List[str]] = Query(None, description="Lista de materiales a filtrar"),
    color: Optional[List[str]] = Query(None, description="Lista de colores a filtrar"),
    caracteristicas: Optional[List[str]] = Query(None, description="Lista de características a filtrar"),
    forma: Optional[List[str]] = Query(None, description="Lista de formas a filtrar"),
    skip: int = Query(0, ge=0),
    limit: int = Query(10, ge=1, le=100)
):
    """Get filtered mobiliari decoracio services for the current user"""
    return crud.mobiliari_decoracio.get_mobiliari_decoracios(
        session=session,
        skip=skip,
        limit=limit,
        service_type=service_type,
        tipologia=tipologia,
        is_rental=is_rental,
        cantidad=cantidad,
        material=material,
        color=color,
        caracteristicas=caracteristicas,
        forma=forma,
        user_id=current_user.id
    ) 