""" Raider routes """
from typing import List, Optional
from uuid import UUID

from fastapi import APIRouter, Depends, HTTPException, Query
from sqlmodel import Session

from app.api.deps import CurrentUser, SessionDep, get_current_active_superuser
from app import crud
from app.models import Raider, RaiderCreate, RaiderUpdate, RaiderOut, RaiderFilter, RaidersOut, RaidersOutId, RoleEnum


router = APIRouter()


@router.post("/", response_model=RaiderOut)
def create_raider(
    raider_in: RaiderCreate,
    session: SessionDep,
    current_user: CurrentUser
) -> RaiderOut:
    """Create a new raider for the current account"""
    if not current_user.is_superuser and not crud.user.is_user_vendor(session=session, user_id=current_user.id):
        raise HTTPException(status_code=403, detail="User must be a vendor or superuser")
    
    return crud.raider.create_raider(
        session=session,
        raider_in=raider_in,
        account_id=current_user.id
    )


@router.get("/by_id/{raider_id}", response_model=RaiderOut)
def get_raider(
    raider_id: UUID,
    session: SessionDep,
) -> Raider:
    """Get a raider by ID"""
    raider = crud.raider.get_raider(
        session=session,
        raider_id=raider_id
    )
    if not raider:
        raise HTTPException(status_code=404, detail="Raider not found")
    return raider


@router.get("/all", response_model=RaidersOut, dependencies=[Depends(get_current_active_superuser)])
def get_all_raiders(
    session: SessionDep,
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=100)
) -> RaidersOut:
    """Get all raiders (admin only)"""
    return crud.raider.get_all_raiders(
        session=session,
        skip=skip,
        limit=limit
    )



@router.get("/me", response_model=RaidersOut)
def get_my_raiders(
    session: SessionDep,
    current_user: CurrentUser,
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=100)
) -> RaidersOut:
    """Get all raiders for the current user"""
    return crud.raider.get_raiders_by_account(
        session=session,
        account_id=current_user.id,
        skip=skip,
        limit=limit
    )


@router.get("/filter", response_model=RaidersOut)
def get_filtered_raiders(
    session: SessionDep,
    current_user: CurrentUser,
    solicitar_cancion_nueva: Optional[bool] = None,
    equipo_propio: Optional[bool] = None,
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=100)
) -> RaidersOut:
    """Get filtered raiders"""
    filters = RaiderFilter(
        solicitar_cancion_nueva=solicitar_cancion_nueva,
        equipo_propio=equipo_propio
    )
    return crud.raider.get_filtered_raiders(
        session=session,
        filters=filters,
        skip=skip,
        limit=limit
    )


@router.patch("/{raider_id}", response_model=RaiderOut)
def update_raider(
    raider_id: UUID,
    raider_in: RaiderUpdate,
    session: SessionDep,
    current_user: CurrentUser
) -> RaiderOut:
    """Update a raider"""
    if not current_user.is_superuser and not crud.user.is_user_vendor(session=session, user_id=current_user.id):
        raise HTTPException(status_code=403, detail="User must be a vendor or superuser")

    raider = crud.raider.get_raider(
        session=session,
        raider_id=raider_id
    )
    if not raider:
        raise HTTPException(status_code=404, detail="Raider not found")
    
    if not current_user.is_superuser and raider.account_id != current_user.id:
        raise HTTPException(status_code=403, detail="User can only update their own raiders")
    
    return crud.raider.update_raider(
        session=session,
        raider=raider,
        raider_in=raider_in
    )


@router.put("/{raider_id}", response_model=RaiderOut)
def replace_raider(
    raider_id: UUID,
    raider_in: RaiderCreate,
    session: SessionDep,
    current_user: CurrentUser
) -> Raider:
    """Replace a raider completely"""
    if not current_user.is_superuser and not crud.user.is_user_vendor(session=session, user_id=current_user.id):
        raise HTTPException(status_code=403, detail="User must be a vendor or superuser")

    raider = crud.raider.get_raider(
        session=session,
        raider_id=raider_id
    )
    if not raider:
        raise HTTPException(status_code=404, detail="Raider not found")
    
    if not current_user.is_superuser and raider.account_id != current_user.id:
        raise HTTPException(status_code=403, detail="User can only replace their own raiders")
    
    return crud.raider.replace_raider(
        session=session,
        raider=raider,
        raider_in=raider_in
    )


@router.delete("/{raider_id}")
def delete_raider(
    raider_id: UUID,
    session: SessionDep,
    current_user: CurrentUser
) -> Raider:
    """Delete a raider"""
    if not current_user.is_superuser and not crud.user.is_user_vendor(session=session, user_id=current_user.id):
        raise HTTPException(status_code=403, detail="User must be a vendor or superuser")

    raider = crud.raider.get_raider(
        session=session,
        raider_id=raider_id
    )
    if not raider:
        raise HTTPException(status_code=404, detail="Raider not found")
    
    if not current_user.is_superuser and raider.account_id != current_user.id:
        raise HTTPException(status_code=403, detail="User can only delete their own raiders")
    
    exit = crud.raider.delete_raider(
        session=session,
        raider=raider
    ) 
    if not exit:
        raise HTTPException(status_code=404, detail="Raider not deleted")
    return {"message": "Raider deleted successfully"}