""" Service routes """
from typing import List, Optional, Dict, Any
from uuid import UUID

from fastapi import APIRouter, Depends, HTTPException, Query
from sqlmodel import Session

from app.api.deps import CurrentUser, SessionDep, get_current_active_superuser
from app import crud
from app.models import (
    ServiceOut, ServicesOut, ServiceType
)


router = APIRouter()


@router.get("/all", response_model=ServicesOut)
def get_all_services(
    session: SessionDep,
    skip: int = Query(0, ge=0),
    limit: int = Query(200, ge=1, le=200)
) -> ServicesOut:
    """Get all services (admin only)"""
    return crud.service.get_all_services(
        session=session,
        skip=skip,
        limit=limit
    )


@router.get("/me", response_model=ServicesOut)
def get_my_services(
    session: SessionDep,
    current_user: CurrentUser,
    skip: int = Query(0, ge=0),
    limit: int = Query(200, ge=1, le=200)
) -> ServicesOut:
    """Get all services for the current user"""
    return crud.service.get_services_by_account(
        session=session,
        account_id=current_user.id,
        skip=skip,
        limit=limit
    )


@router.get("/by_id/{service_id}")
def get_service(
    service_id: UUID,
    session: SessionDep,
) -> Dict[str, Any]:
    """Get a service by ID"""
    service = crud.service.get_service(
        session=session,
        service_id=service_id
    )
    if not service:
        raise HTTPException(status_code=404, detail="Service not found")
    return service


@router.get("/filter", response_model=ServicesOut)
def get_filtered_services(
    session: SessionDep,
    type: Optional[ServiceType] = None,
    name: Optional[str] = None,
    provincia: Optional[str] = None,
    ciudad: Optional[str] = None,
    dia_semana: Optional[int] = Query(None, ge=1, le=7, description="Día de la semana (1-7)"),
    precio_min: Optional[float] = Query(None, ge=0, description="Precio mínimo"),
    precio_max: Optional[float] = Query(None, ge=0, description="Precio máximo"),
    skip: int = Query(0, ge=0),
    limit: int = Query(200, ge=1, le=200)
) -> ServicesOut:
    """Get filtered services"""
    return crud.service.get_filtered_services(
        session=session,
        filters={
            "type": type,
            "name": name,
            "provincia": provincia,
            "ciudad": ciudad,
            "dia_semana": dia_semana,
            "precio_min": precio_min,
            "precio_max": precio_max
        },
        skip=skip,
        limit=limit
    )


@router.get("/by_raider/{raider_id}", response_model=ServicesOut)
def get_services_by_raider(
    raider_id: UUID,
    session: SessionDep,
) -> ServicesOut:
    """
    Get all services (fotografía o música) vinculados a un raider
    """
    return crud.service.get_services_by_raider(session=session, raider_id=raider_id)


@router.get("/by_account/{account_id}", response_model=ServicesOut)
def get_services_by_account_id(
    account_id: UUID,
    session: SessionDep,
    skip: int = Query(0, ge=0),
    limit: int = Query(200, ge=1, le=200)
) -> ServicesOut:
    """
    Get all services for a specific account by ID
    """
    return crud.service.get_services_by_account(
        session=session,
        account_id=account_id,
        skip=skip,
        limit=limit
    ) 