""" Account related CRUD methods """
from typing import Any

from sqlmodel import Session, select

from fastapi import HTTPException

from app.models import Account, AccountCreate, AccountUpdate, Message, User
import uuid

def create_account(session: Session, account: AccountCreate) -> Account:
    db_obj = Account(
        id=account.id,
        photo=account.photo,
        bio=account.bio,
        username=account.username,
    )
    session.add(db_obj)
    session.commit()
    session.refresh(db_obj)
    return db_obj

def update_account(*, session: Session, db_account: Account, account_in: AccountUpdate) -> Any:
    account_data = account_in.model_dump(exclude_unset=True)
    db_account.sqlmodel_update(account_data)
    session.add(db_account)
    session.commit()
    session.refresh(db_account)
    return db_account

def get_account(*, session: Session, account_id: uuid.UUID) -> Account | None:
    statement = select(Account).where(Account.id == account_id)
    session_account = session.exec(statement).first()
    return session_account

def delete_account(*, session: Session, account: Account, current_user: User) -> Message:
    if not account:
        raise HTTPException(status_code=404, detail="Account not found")

    elif current_user.is_superuser:
        raise HTTPException(
            status_code=403, detail="Super users are not allowed to delete themselves"
        )

    session.delete(account)
    session.commit()
    return Message(message="Account deleted successfully")
    