""" Account models """
import uuid
from datetime import datetime
from typing import List, Optional

from sqlmodel import Field, Relationship
from .base import SQLModel

class AccountBase(SQLModel):
    """Modelo base para cuentas"""
    id: uuid.UUID = Field(default=None, primary_key=True, foreign_key="user.id", description="ID único de la cuenta")
    username: str = Field(max_length=255, description="Nombre de usuario")
    photo: str | None = Field(default=None, description="URL de la foto de perfil")
    bio: str | None = Field(default=None, max_length=200, description="Biografía del usuario")

# Database model
class Account(AccountBase, table=True):
    """Modelo de base de datos para cuentas"""
    user: "User" = Relationship(back_populates="account")
    events: List["Event"] = Relationship(back_populates="account")
    services: List["Service"] = Relationship(back_populates="account")
    raiders: List["Raider"] = Relationship(back_populates="account")
    espais_propis: List["EspaiPropi"] = Relationship(back_populates="account")

class AccountUpdate(SQLModel):
    """Modelo para actualizar cuentas"""
    username: str | None = Field(default=None, description="Nombre de usuario")
    photo: str | None = Field(default=None, description="URL de la foto de perfil")
    bio: str | None = Field(default=None, max_length=200, description="Biografía del usuario")

class AccountUpdateMe(SQLModel):
    """Modelo para actualizar la propia cuenta"""
    username: str | None = Field(default=None, description="Nombre de usuario")
    photo: str | None = Field(default=None, description="URL de la foto de perfil")
    bio: str | None = Field(default=None, max_length=200, description="Biografía del usuario")

class AccountCreate(AccountBase):
    """Modelo para crear cuentas"""
    pass

class AccountOut(AccountBase):
    """Modelo de salida para cuentas"""
    pass

class AccountsOut(SQLModel):
    """Modelo de salida para listas de cuentas"""
    data: List[AccountOut] = Field(description="Lista de cuentas")
    count: int = Field(description="Número total de cuentas")

class AccountUserOut(SQLModel):
    """Modelo de salida para cuentas de usuario"""
    id: uuid.UUID = Field(description="ID único de la cuenta")
    username: str = Field(description="Nombre de usuario")
    photo: str | None = Field(default=None, description="URL de la foto de perfil")
    bio: str | None = Field(default=None, max_length=200, description="Biografía del usuario")
    role: str = Field(description="Rol del usuario")
    full_name: str = Field(description="Nombre completo del usuario")
    email: str = Field(description="Correo electrónico del usuario")

class AccountsUserOut(SQLModel):
    """Modelo de salida para listas de cuentas de usuario"""
    data: List[AccountUserOut] = Field(description="Lista de cuentas de usuario")
    count: int = Field(description="Número total de cuentas de usuario")
