""" Availability models """
from typing import List, Optional
from uuid import UUID, uuid4
from sqlmodel import SQLModel, Field, Relationship, Column, JSON
from datetime import time

class AvailabilityRuleBase(SQLModel):
    """Base model for availability rules"""
    dias_semana: List[int] = Field(default_factory=list, sa_column=Column(JSON), description="Días de la semana, e.g., [3, 4, 5] = miércoles, jueves, viernes")
    hora_inicio: time = Field(description="Hora de inicio")
    hora_fin: time = Field(description="Hora de fin")
    price: float = Field(description="Precio del servicio para esta disponibilidad")
    min_days_before: int = Field(default=1, description="Días mínimos antes de la fecha de inicio del evento")

class AvailabilityRule(AvailabilityRuleBase, table=True):
    """Database model for availability rules"""
    id: UUID = Field(default_factory=uuid4, primary_key=True)
    service_id: UUID = Field(foreign_key="service.id", description="Referencia al servicio asociado")
    service: "Service" = Relationship(back_populates="availability_rules")

class AvailabilityRuleCreate(AvailabilityRuleBase):
    """Model for creating availability rules"""
    pass

class AvailabilityRuleUpdate(SQLModel):
    """Model for updating availability rules"""
    dias_semana: Optional[List[int]] = Field(default=None, sa_column=Column(JSON), description="Días de la semana")
    hora_inicio: Optional[time] = Field(default=None, description="Hora de inicio")
    hora_fin: Optional[time] = Field(default=None, description="Hora de fin")
    price: Optional[float] = Field(default=None, description="Precio del servicio para esta disponibilidad")

class AvailabilityRuleOut(AvailabilityRuleBase):
    """Output model for availability rules"""
    id: UUID 
    service_id: UUID