from pydantic import BaseModel, root_validator, model_validator
from typing import List, Optional, Dict, Any
from datetime import datetime
from .espai_propi import EspaiPropi
import uuid

class Message(BaseModel):
    """Modelo para un mensaje individual en la conversación"""
    text: str
    type: str  # 'user' o 'assistant'
    timestamp: datetime = datetime.now()

class ChatHistory(BaseModel):
    """Modelo para el historial de chat"""
    messages: List[Message] = []
    event_id: Optional[uuid.UUID] = None
    user_id: uuid.UUID

class EventData(BaseModel):
    name: str
    tipus: str
    pressupost: float
    data_inici: datetime
    data_fi: datetime
    num_invitats: int
    espai_propi_id: Optional[uuid.UUID] = None
    espai_propi: Dict = {}

class ChatRequest(BaseModel):
    event: EventData
    services: List[str]
    message: str
    context: str
class ChatResponse(BaseModel):
    needs_more_info: bool = False
    message: str
    selected_services: List[Any]

class Assistent(BaseModel):
    """Modelo de entrada para el asistente"""
    event_id: uuid.UUID
    serveis: Dict[str, bool]  # ej: {"Espai": true, "Catering": false, ...}
    message: str
    chat_history: Optional[List[Message]] = None

    @model_validator(mode='after')
    def validate_services(self) -> 'Assistent':
        required_services = ["Espai", "Catering", "Mobiliari", "Decoracio", "Musica", "Fotografia"]
        for service in required_services:
            if service not in self.serveis:
                self.serveis[service] = False
        return self

"""
class AssistenteInput(BaseModel):
    tipus_event: str  # ej: "casament", "aniversari"
    serveis: Dict[str, bool]  # ej: {"Espai": true, "Catering": false, ...}
    espai_propi: Optional[EspaiPropi] = None
    datetime_inicio: datetime
    datetime_fin: Optional[datetime] = None
    duracion_horas: Optional[int] = None
    num_personas: int
    presupost: float
    especificacions: Optional[str] = None
    
    @model_validator(mode='after')
    def validar_entrada(self) -> 'AssistenteInput':
        espai_propi = self.espai_propi
        serveis = self.serveis
        inicio = self.datetime_inicio
        fin = self.datetime_fin
        duracion = self.duracion_horas

        # Validar que se da solo fin o duración
        if (fin is None and duracion is None) or (fin is not None and duracion is not None):
            raise ValueError("Debes proporcionar solo uno: 'datetime_fin' o 'duracion_horas', pero no ambos.")

        # Validar campo Espai
        if "Espai" not in serveis:
            raise ValueError("El campo 'Espai' debe estar en la lista de serveis.")

        if espai_propi is None and not serveis.get("Espai", False):
            raise ValueError("Si no se proporciona espai_propi, 'Espai' debe ser True.")
        if espai_propi is not None and serveis.get("Espai", False):
            raise ValueError("Si se proporciona espai_propi, 'Espai' debe ser False.")

        return self

class AssistenteOutput(BaseModel):
    """ """Modelo de salida para el asistente""" """
    message: str
    services: List[Dict[str, Any]] 
    
"""