""" Modelo para espacios propios """
import uuid
from typing import List, Optional
from enum import Enum
from sqlmodel import Field, Relationship, Column, JSON
from .base import SQLModel
from .service import Address

class TipusEspaiPropi(str, Enum):
    """Tipos de espacios propios disponibles"""
    SALA_PRIVADA = "sala_privada"
    FINCA = "finca"
    TERRAZA = "terraza"
    SALON = "salon"
    CASA_PARTICULAR = "casa_particular"
    JARDIN_PRIVADO = "jardin_privado"
    AZOTEA = "azotea"
    PISO = "piso"
    CASA_RURAL = "casa_rural"
    NAVE_INDUSTRIAL = "nave_industrial"
    PARCELA = "parcela"
    LOCAL_PROPIO = "local_propio"
    MASIA = "masia"
    OTRO = "otro"

class EspaiPropiBase(SQLModel):
    """Modelo base para espacios propios"""
    nombre: str = Field(max_length=255, description="Nombre del espacio propio")
    tipus: TipusEspaiPropi = Field(description="Tipo de espacio propio")
    address: Address = Field(sa_column=Column(JSON), description="Dirección del espacio propio")
    descripcion: Optional[str] = Field(default=None, description="Descripción del espacio propio")
    capacitat_aproximada: Optional[int] = Field(default=None, description="Capacidad aproximada del espacio")
    caracteristicas: List[str] = Field(default_factory=list, sa_column=Column(JSON), description="Lista de características del espacio")

# Database model
class EspaiPropi(EspaiPropiBase, table=True):
    """Modelo de base de datos para espacios propios"""
    __tablename__ = "espai_propi"  # Añadimos explícitamente el nombre de la tabla
    id: uuid.UUID = Field(default_factory=uuid.uuid4, primary_key=True)
    account_id: uuid.UUID = Field(foreign_key="account.id")
    account: "Account" = Relationship(back_populates="espais_propis")
    events: List["Event"] = Relationship(back_populates="espai_propi")

# Out models
class EspaiPropiOut(EspaiPropiBase):
    """Modelo para respuestas de espacios propios"""
    id: uuid.UUID
    account_id: uuid.UUID

class EspaiPropisOut(SQLModel):
    """Modelo para listas de espacios propios"""
    data: List[EspaiPropiOut]
    count: int
    

# Create/Update models
class EspaiPropiCreate(EspaiPropiBase):
    """Modelo para crear espacios propios"""
    pass

class EspaiPropiUpdate(SQLModel):
    """Modelo para actualizar espacios propios"""
    nombre: Optional[str] = Field(default=None, max_length=255, description="Nombre del espacio propio")
    tipus: Optional[TipusEspaiPropi] = Field(default=None, description="Tipo de espacio propio")
    address: Optional[Address] = Field(default=None, sa_column=Column(JSON), description="Dirección del espacio propio")
    descripcion: Optional[str] = Field(default=None, description="Descripción del espacio propio")
    capacitat_aproximada: Optional[int] = Field(default=None, description="Capacidad aproximada del espacio")
    caracteristicas: Optional[List[str]] = Field(default=None, sa_column=Column(JSON), description="Lista de características del espacio")