""" Fotografia service models """
import uuid
from typing import List, Optional, Dict, Any
from enum import Enum

from sqlmodel import Field, Relationship, Column, JSON
from .base import SQLModel


class FotografiaType(str, Enum):
    """Estilos de servicio de fotografía disponibles"""
    FOTOGRAFIA = "fotografia"
    VIDEO = "video"
    SERVICIO_ESPECIAL = "servicio_especial"


class ServicioEspecial(str, Enum):
    """Tipos de servicios especiales de fotografía disponibles"""
    IMPRESION_INSTANTANEA = "impresion_instantanea"
    FOTOMATON = "fotomaton"
    CABINA = "cabina"
    ESPEJO_MAGICO = "espejo_magico"
    VIDEO_360 = "video_360"
    GIFT_BOX = "gift_box"
    VIRTUAL = "virtual"
    CAMARA_OCULTA = "camara_oculta"


class FotografiaBase(SQLModel):
    """Modelo base para servicios de fotografía"""
    fotografia_type: FotografiaType = Field(description="Tipo de servicio (fotógrafo o videógrafo)")
    servicio_especial_type: Optional[ServicioEspecial] = Field(default=None, description="Tipo de servicio especial si aplica")
    duracion: float = Field(description="Duración del servicio en horas")
    caracteristicas: List[str] = Field(default_factory=list, sa_column=Column(JSON), description="Características especiales del servicio")


# Database model
class Fotografia(FotografiaBase, table=True):
    """Modelo de base de datos para servicios de fotografía"""
    id: uuid.UUID = Field(default=None, primary_key=True, foreign_key="service.id", description="ID único del servicio de fotografía")
    service: "Service" = Relationship(back_populates="fotografia")
    raider_id: uuid.UUID = Field(foreign_key="raider.id")
    raider: "Raider" = Relationship(back_populates="fotografias")


# Out models
class FotografiaOut(FotografiaBase):
    """Modelo de salida para servicios de fotografía"""
    id: uuid.UUID
    raider_id: uuid.UUID


class FotografiasOut(SQLModel):
    """Modelo de salida para listas de servicios de fotografía"""
    data: List[FotografiaOut] = Field(description="Lista de servicios de fotografía")
    count: int = Field(description="Número total de servicios")



class FotografiaServiceOut(FotografiaBase):
    """Modelo de salida para un servicio de fotografía completo con su raider"""
    id: uuid.UUID
    raider_id: uuid.UUID
    raider: Dict[str, Any] = Field(default=None, sa_column=Column(JSON))

class FotografiasServiceOut(SQLModel):
    """Modelo de salida para listas de servicios de fotografía con su raider"""
    data: List[FotografiaServiceOut] = Field(description="Lista de servicios de fotografía con su raider")
    count: int = Field(description="Número total de servicios de fotografía con su raider")


# Create/Update models
class FotografiaCreate(FotografiaBase):
    """Modelo para crear servicios de fotografía"""
    pass


class FotografiaUpdate(SQLModel):
    """Modelo para actualizar servicios de fotografía"""
    fotografia_type: Optional[FotografiaType] = Field(default=None, description="Tipo de servicio (fotógrafo o videógrafo)")
    servicio_especial_type: Optional[ServicioEspecial] = Field(default=None, description="Tipo de servicio especial si aplica")
    duracion: Optional[float] = Field(default=None, description="Duración del servicio en horas")
    caracteristicas: Optional[List[str]] = Field(default=None, sa_column=Column(JSON), description="Características especiales del servicio")

class FotografiaServiceCreate(FotografiaBase):
    """Modelo para crear servicios de fotografía"""
    raider: Dict[str, Any] = Field(default=None, sa_column=Column(JSON))

class FotografiaServiceUpdate(SQLModel):
    """Modelo para actualizar servicios de fotografía"""

    fotografia_type: Optional[FotografiaType] = Field(default=None, description="Tipo de servicio (fotógrafo o videógrafo)")
    servicio_especial_type: Optional[ServicioEspecial] = Field(default=None, description="Tipo de servicio especial si aplica")
    duracion: Optional[float] = Field(default=None, description="Duración del servicio en horas")
    caracteristicas: Optional[List[str]] = Field(default=None, sa_column=Column(JSON), description="Características especiales del servicio")
    raider: Optional[Dict[str, Any]] = Field(default=None, sa_column=Column(JSON))
