""" Mobiliari i Decoracio service models """
import uuid
from typing import List, Optional, Dict, Any
from enum import Enum
from pydantic import validator

from sqlmodel import Field, Relationship, Column, JSON
from .base import SQLModel
from .service import ServiceOut

class MobiliariDecoracioType(str, Enum):
    """Tipos de servicio de mobiliario y decoración"""
    DECORACION = "decoracion"
    MOBILIARIO = "mobiliario"


class TipoMobiliriDecoracio(str, Enum):
    """Tipologías específicas de mobiliario y decoración"""
    FLORES = "Flores"
    GLOBOS = "Globos"
    LUCES_GUINALDA = "Luces guirnalda"
    LUCES_LED = "Luces LED"
    FUNDAS_SILLA = "Fundas de silla"
    COJINES_MANTAS = "Cojines y mantas"
    TELAS = "Telas"
    CARTELES = "Carteles"
    PHOTOCALL = "Photocall"
    DISFRACES = "Disfraces"
    VASOS = "Vasos"
    VAJILLA_BANDEJAS = "Vajilla y bandejas"
    MANTELES_SERVILLETAS = "Manteles y servilletas"
    UTENSILIOS_COCINA = "Utensilios de cocina"
    ELECTRODOMESTICOS_COCINA = "Electrodomésticos y maquinaria de cocina"
    BUFET = "Bufet"
    FOODTRUCK = "Foodtruck"
    BARBACOA = "Barbacoa"
    MESAS = "Mesas"
    SILLAS_TABURETES = "Sillas y taburetes"
    TRONAS = "Tronas"
    BARRILES_TONELADAS_CHILL_OUT = "Barras, barriles, toneles, chill-out"
    ESCALERAS_VIGAS = "Escaleras y vigas"
    PALES = "Pales"
    CATENARIAS = "Catenarias"
    COLGADORES_GUARDARROPAS = "Colgadores y guardarropas"
    CUBIERTA = "Cubierta"
    CRISTALERA = "Cristalera"
    CARPAS = "Carpas"
    TOLDOS_PARASOLES = "Toldos y parasoles"
    PAREDES = "Paredes"
    SUELOS = "Suelos"
    EDIFICACIONES_TEMPORALES = "Edificaciones temporales"
    STANDS = "Stands"
    TOCADOR_MAQUILLAJE = "Tocador maquillaje"
    ESPEJOS = "Espejos"
    VITRINAS = "Vitrinas"
    MOSTRADORES_BARRAS = "Mostradores y barras"
    IGLOO = "Iglú"
    ESTUFAS_CALEFACTORES = "Estufas y calefactores"
    VENTILADORES_AIRE = "Ventiladores y aire acondicionado"
    PIZARRAS = "Pizarras"
    PANTALLAS = "Pantallas"
    PROYECTORES = "Proyectores"
    ALTAVOCES = "Altavoces"
    MICROFONOS = "Micrófonos"
    FOCO = "Focos"
    BOLA_DISCO = "Bola de discoteca"
    TABLA_DJ = "Tabla DJ"
    ELECTRICIDAD_ENCHUFES = "Electricidad y enchufes"
    AIRES_FAROLILLOS = "Aires y farolillos"
    LIMPIEZA = "Limpieza"
    GRADAS_TRIBUNAS = "Gradas y tribunas"
    JUEGOS = "Juegos"
    ATRACCIONES = "Atracciones"
    MANIQUIS = "Maniquíes"
    MAQUINAS_COMIDA = "Máquinas de comida"
    WC = "WC"
    CONTENEDORES_PAPELERAS = "Contenedores y papeleras"


class Dimensions(SQLModel):
    """Modelo para las dimensiones físicas de un elemento"""
    width: float = Field(description="Ancho en unidades de medida")
    height: float = Field(description="Alto en unidades de medida")
    depth: float = Field(description="Profundidad en unidades de medida")



class MobiliariDecoracioBase(SQLModel):
    """Modelo base para mobiliario y decoración"""
    service_type: MobiliariDecoracioType = Field(description="Tipo de servicio (decoración o mobiliario)")
    tipologia: TipoMobiliriDecoracio = Field(description="Tipología específica del elemento")
    is_rental: bool = Field(default=True, description="Indica si el elemento es para alquiler")
    cantidad_min: int = Field(description="Cantidad mínima disponible")
    cantidad_max: int = Field(description="Cantidad máxima disponible")
    subtipologia: Optional[str] = Field(default=None, description="Subtipología del elemento")
    forma: Optional[str] = Field(default=None, description="Forma del elemento")
    modality: Optional[str] = Field(default=None, description="Modalidad de uso")
    color: List[str] = Field(default_factory=list, sa_column=Column(JSON), description="Colores disponibles")
    material: List[str] = Field(default_factory=list, sa_column=Column(JSON), description="Materiales del elemento")
    style: Optional[str] = Field(default=None, description="Estilo del elemento")
    size: Optional[str] = Field(default=None, description="Tamaño descriptivo del elemento")
    dimensions: Optional[Dict[str, float]] = Field(default=None, sa_column=Column(JSON), description="Dimensiones del elemento")
    caracteristicas: List[str] = Field(default_factory=list, sa_column=Column(JSON), description="Características especiales")
    complement: Optional[str] = Field(default=None, description="Complementos incluidos")

    @validator('cantidad_max')
    def validate_cantidad_max(cls, v, values):
        """Valida que cantidad_max sea mayor o igual que cantidad_min"""
        if 'cantidad_min' in values and v < values['cantidad_min']:
            raise ValueError('cantidad_max debe ser mayor o igual que cantidad_min')
        return v


# Database model
class MobiliariDecoracio(MobiliariDecoracioBase, table=True):
    """Modelo de base de datos para mobiliario y decoración"""
    id: uuid.UUID = Field(default=None, primary_key=True, foreign_key="service.id", description="ID único del servicio de mobiliario y decoración")
    service: "Service" = Relationship(back_populates="mobiliari_decoracio")


# Out models
class MobiliariDecoracioOut(MobiliariDecoracioBase):
    """Modelo de salida para mobiliario y decoración"""
    id: uuid.UUID

class MobiliariDecoraciosOut(SQLModel):
    """Modelo de salida para listas de mobiliario y decoración"""
    data: List[MobiliariDecoracioOut] = Field(description="Lista de mobiliario y decoración")
    count: int = Field(description="Número total de elementos")


class MobiliariDecoracioServiceOut(MobiliariDecoracioBase):
    """Modelo de salida para mobiliario y decoración"""
    id: uuid.UUID


class MobiliariDecoraciosServiceOut(SQLModel):
    """Modelo de salida para listas de mobiliario y decoración"""
    data: List[MobiliariDecoracioServiceOut] = Field(description="Lista de mobiliario y decoración")
    count: int = Field(description="Número total de elementos")


# Create/Update models
class MobiliariDecoracioCreate(MobiliariDecoracioBase):
    """Modelo para crear mobiliario y decoración"""
    pass


class MobiliariDecoracioUpdate(SQLModel):
    """Modelo para actualizar mobiliario y decoración"""
    service_type: Optional[MobiliariDecoracioType] = Field(default=None, description="Tipo de servicio (decoración o mobiliario)")
    tipologia: Optional[TipoMobiliriDecoracio] = Field(default=None, description="Tipología específica del elemento")
    is_rental: Optional[bool] = Field(default=None, description="Indica si el elemento es para alquiler")
    cantidad_min: Optional[int] = Field(default=None, description="Cantidad mínima disponible")
    cantidad_max: Optional[int] = Field(default=None, description="Cantidad máxima disponible")
    subtipologia: Optional[str] = Field(default=None, description="Subtipología del elemento")
    forma: Optional[str] = Field(default=None, description="Forma del elemento")
    modality: Optional[str] = Field(default=None, description="Modalidad de uso")
    color: Optional[List[str]] = Field(default=None, sa_column=Column(JSON), description="Colores disponibles")
    material: Optional[List[str]] = Field(default=None, sa_column=Column(JSON), description="Materiales del elemento")
    style: Optional[str] = Field(default=None, description="Estilo del elemento")
    size: Optional[str] = Field(default=None, description="Tamaño descriptivo del elemento")
    caracteristicas: Optional[List[str]] = Field(default=None, sa_column=Column(JSON), description="Características especiales")
    complement: Optional[str] = Field(default=None, description="Complementos incluidos")

    @validator('cantidad_max')
    def validate_cantidad_max(cls, v, values):
        """Valida que cantidad_max sea mayor o igual que cantidad_min"""
        if 'cantidad_min' in values and v is not None and values['cantidad_min'] is not None and v < values['cantidad_min']:
            raise ValueError('cantidad_max debe ser mayor o igual que cantidad_min')
        return v
    

class MobiliariDecoracioServiceCreate(MobiliariDecoracioBase):
    """Modelo para crear mobiliario y decoración"""
    pass


class MobiliariDecoracioServiceUpdate(SQLModel):
    """Modelo para actualizar mobiliario y decoración"""
    service_type: Optional[MobiliariDecoracioType] = Field(default=None, description="Tipo de servicio (decoración o mobiliario)")
    tipologia: Optional[TipoMobiliriDecoracio] = Field(default=None, description="Tipología específica del elemento")
    is_rental: Optional[bool] = Field(default=None, description="Indica si el elemento es para alquiler")
    cantidad_min: Optional[int] = Field(default=None, description="Cantidad mínima disponible")
    cantidad_max: Optional[int] = Field(default=None, description="Cantidad máxima disponible")
    subtipologia: Optional[str] = Field(default=None, description="Subtipología del elemento")
    forma: Optional[str] = Field(default=None, description="Forma del elemento")
    modality: Optional[str] = Field(default=None, description="Modalidad de uso")
    color: Optional[List[str]] = Field(default=None, sa_column=Column(JSON), description="Colores disponibles")
    material: Optional[List[str]] = Field(default=None, sa_column=Column(JSON), description="Materiales del elemento")
    style: Optional[str] = Field(default=None, description="Estilo del elemento")
    size: Optional[str] = Field(default=None, description="Tamaño descriptivo del elemento")
    caracteristicas: Optional[List[str]] = Field(default=None, sa_column=Column(JSON), description="Características especiales")
    complement: Optional[str] = Field(default=None, description="Complementos incluidos")

    @validator('cantidad_max')
    def validate_cantidad_max(cls, v, values):
        """Valida que cantidad_max sea mayor o igual que cantidad_min"""
        if 'cantidad_min' in values and v is not None and values['cantidad_min'] is not None and v < values['cantidad_min']:
            raise ValueError('cantidad_max debe ser mayor o igual que cantidad_min')
        return v
