""" Raider models """
import uuid
from typing import List, Optional, Dict

from sqlmodel import Field, Relationship, Column, JSON
from .base import SQLModel


class RaiderBase(SQLModel):
    """Modelo base para raiders""" 
    num_integrantes: int = Field(description="Número de integrantes del equipo")
    equipo: List[Dict[str, str]] = Field(default_factory=list, sa_column=Column(JSON), description="Lista de miembros del equipo con nombre y rol")
    solicitar_cancion_nueva: Optional[bool] = Field(default=None, description="Indica si se pueden solicitar canciones nuevas")
    equipo_propio: Optional[bool] = Field(default=None, description="Indica si se requiere equipo propio")
    dimensiones_escenario: Optional[Dict[str, float]] = Field(default_factory=dict, sa_column=Column(JSON), description="Dimensiones requeridas para el escenario")
    condiciones: List[str] = Field(default_factory=list, sa_column=Column(JSON), description="Condiciones especiales del servicio")


# Database model
class Raider(RaiderBase, table=True):
    """Modelo de base de datos para raiders"""
    id: uuid.UUID = Field(default_factory=uuid.uuid4, primary_key=True)
    musicas: List["Musica"] = Relationship(back_populates="raider")
    fotografias: List["Fotografia"] = Relationship(back_populates="raider")
    account_id: uuid.UUID = Field(foreign_key="account.id")
    account: "Account" = Relationship(back_populates="raiders")


# Out models
class RaiderOut(RaiderBase):
    """Modelo para respuestas de raiders"""
    id: uuid.UUID
    account_id: uuid.UUID

class RaidersOut(SQLModel):
    """Modelo para listas de raiders"""
    data: List[RaiderOut]
    count: int


class RaiderOutId(SQLModel):
    """Modelo para respuestas de IDs de raiders"""
    id: uuid.UUID
    account_id: uuid.UUID


class RaidersOutId(SQLModel):
    """Modelo para listas de IDs de raiders"""
    data: List[RaiderOutId]
    count: int


# Create/Update models
class RaiderCreate(RaiderBase):
    """Modelo para crear raiders"""
    pass


class RaiderUpdate(SQLModel):
    """Modelo para actualizar raiders"""
    num_integrantes: Optional[int] = Field(default=None, description="Número de integrantes del equipo")
    equipo: Optional[List[Dict[str, str]]] = Field(default=None, sa_column=Column(JSON), description="Lista de miembros del equipo con nombre y rol")
    solicitar_cancion_nueva: Optional[bool] = Field(default=None, description="Indica si se pueden solicitar canciones nuevas")
    coste_cancion_nueva: Optional[float] = Field(default=None, description="Coste por canción nueva")
    equipo_propio: Optional[bool] = Field(default=None, description="Indica si se requiere equipo propio")
    coste_equipo_propio: Optional[float] = Field(default=None, description="Coste por uso de equipo propio")
    disponible_sin_equipo: Optional[bool] = Field(default=None, description="Indica si está disponible sin equipo")
    dimensiones_escenario: Optional[Dict[str, float]] = Field(default=None, sa_column=Column(JSON), description="Dimensiones requeridas para el escenario")
    condiciones: Optional[List[str]] = Field(default=None, sa_column=Column(JSON), description="Condiciones especiales del servicio")


class RaiderFilter(SQLModel):
    """Modelo para filtrar raiders"""
    solicitar_cancion_nueva: Optional[bool] = None
    equipo_propio: Optional[bool] = None