""" Event-Service relationship models """
import uuid
from datetime import datetime, timezone
from sqlmodel import SQLModel, Field, Relationship
from typing import List, Optional, Dict, Any


class EventServiceBase(SQLModel):
    """Base model for Event-Service relationship"""
    cantidad: int = Field(default=1, description="Cantidad de servicios contratados")
    data_inicio: datetime = Field(default=datetime.now(timezone.utc), description="Fecha y hora de inicio del servicio")
    data_fin: datetime = Field(default=datetime.now(timezone.utc), description="Fecha y hora de fin del servicio")



class EventService(EventServiceBase, table=True):
    """Database model for Event-Service relationship"""
    id: uuid.UUID = Field(default_factory=uuid.uuid4, primary_key=True)
    event_id: uuid.UUID = Field(foreign_key="event.id")
    service_id: uuid.UUID = Field(foreign_key="service.id")


class EventServiceCreate(EventServiceBase):
    """Model for creating Event-Service relationship"""
    event_id: uuid.UUID
    service_id: uuid.UUID


class EventServiceUpdate(SQLModel):
    """Model for updating Event-Service relationship"""
    cantidad: Optional[int] = Field(default=None, description="Cantidad de servicios contratados")
    data_inicio: Optional[datetime] = Field(default=None)
    data_fin: Optional[datetime] = Field(default=None)


class EventServiceOut(SQLModel):
    """Output model for Event-Service relationship"""
    id: uuid.UUID
    event_id: uuid.UUID
    cantidad: int
    data_inicio: datetime
    data_fin: datetime
    price_total: Optional[float] = Field(default=None, description="Precio total del servicio")
    service: Dict[str, Any]

class EventServicesOut(SQLModel):
    """Output model for list of Event-Service relationships"""
    data: List[EventServiceOut]
    count: int
