from fastapi import APIRouter, Depends, HTTPException
from sqlalchemy.orm import Session
from typing import Optional, List
from pydantic import BaseModel
from ..database import get_db
from ..models.event import EventBase
from ..models.espai_propi import EspaiPropiBase
from ..services.assistant_service import AssistantService

router = APIRouter(
    prefix="/assistant",
    tags=["assistant"]
)

class AssistantRequest(BaseModel):
    message: str
    event_data: Optional[EventBase] = None

class AssistantResponse(BaseModel):
    message: str
    suggest_confirmation: bool
    event_data: Optional[EventBase] = None

class SpaceSelectionRequest(BaseModel):
    use_existing: bool
    space_id: Optional[int] = None
    new_space: Optional[EspaiPropiBase] = None

@router.post("/process-message", response_model=AssistantResponse)
async def process_message(
    request: AssistantRequest,
    db: Session = Depends(get_db)
):
    try:
        assistant_service = AssistantService()
        response = await assistant_service.process_message(request.message, request.event_data)
        return response
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

@router.post("/handle-space-selection")
async def handle_space_selection(request: SpaceSelectionRequest):
    try:
        if request.use_existing:
            if not request.space_id:
                raise HTTPException(status_code=400, detail="Se requiere un ID de espacio cuando se usa un espacio existente")
            # Aquí iría la lógica para validar y obtener el espacio existente
            return {"message": "Espacio existente seleccionado correctamente"}
        else:
            if not request.new_space:
                raise HTTPException(status_code=400, detail="Se requieren los datos del nuevo espacio")
            # Aquí iría la lógica para crear el nuevo espacio
            return {"message": "Nuevo espacio creado correctamente"}
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e)) 