import http from '@/http-common'

class EspaiService {
  // Obtener todos los espacios
  getAllEspais (skip = 0, limit = 10) {
    return http.get(`/api/v1/espai/all/?skip=${skip}&limit=${limit}`)
      .then(res => {
        return res.data
      })
      .catch(error => {
        console.error('Error fetching espai', error)
        throw error
      })
  }

  // Obtener mis espacios
  getAllMyEspais (skip = 0, limit = 10) {
    const token = localStorage.getItem('token')
    return http.get(`/api/v1/espai/all_me/?skip=${skip}&limit=${limit}`, {
      headers: {
        'Authorization': `Bearer ${token}`
      }
    })
      .then(res => {
        return res.data
      })
      .catch(error => {
        console.error('Error fetching my espai', error)
        throw error
      })
  }

  // Obtener un espacio por ID
  getEspaiById (id) {
    return http.get(`/api/v1/espai/by_id/${id}`)
      .then(res => {
        return res.data
      })
      .catch(error => {
        console.error('Error fetching espai', error)
        throw error
      })
  }

  // Crear un nuevo espacio
  createEspai (espaiData) {
    const token = localStorage.getItem('token')
    return http.post('/api/v1/espai/', espaiData, {
      headers: {
        'Authorization': `Bearer ${token}`
      }
    })
      .then(res => {
        return res.data
      })
      .catch(error => {
        console.error('Error creating espai', error)
        throw error
      })
  }

  // Actualizar un espacio
  updateEspai (id, espaiData) {
    const token = localStorage.getItem('token')
    return http.put(`/api/v1/espai/${id}`, espaiData, {
      headers: {
        'Authorization': `Bearer ${token}`
      }
    })
      .then(res => {
        return res.data
      })
      .catch(error => {
        console.error('Error updating espai', error)
        throw error
      })
  }

  // Actualizar parcialmente un espacio
  patchEspai (id, espaiData) {
    const token = localStorage.getItem('token')
    return http.patch(`/api/v1/espai/${id}`, espaiData, {
      headers: {
        'Authorization': `Bearer ${token}`
      }
    })
      .then(res => {
        return res.data
      })
      .catch(error => {
        console.error('Error patching espai', error)
        throw error
      })
  }

  // Eliminar un espacio
  deleteEspai (id) {
    const token = localStorage.getItem('token')
    return http.delete(`/api/v1/espai/${id}`, {
      headers: {
        'Authorization': `Bearer ${token}`
      }
    })
      .then(res => {
        return res.data
      })
      .catch(error => {
        console.error('Error deleting espai', error)
        throw error
      })
  }

  // Filtrar espacios
  filterEspais (params) {
    const queryParams = new URLSearchParams()
    if (params.espai_type) queryParams.append('espai_type', params.espai_type)
    if (params.num_capacitat) queryParams.append('num_capacitat', params.num_capacitat)
    if (params.caracteristicas) {
      params.caracteristicas.forEach(caracteristica => {
        queryParams.append('caracteristicas', caracteristica)
      })
    }
    if (params.skip) queryParams.append('skip', params.skip)
    if (params.limit) queryParams.append('limit', params.limit)

    return http.get(`/api/v1/espai/filter?${queryParams.toString()}`)
      .then(res => {
        return res.data
      })
      .catch(error => {
        console.error('Error filtering espai', error)
        throw error
      })
  }

  // Filtrar mis espacios
  filterMyEspais (params) {
    const token = localStorage.getItem('token')
    const queryParams = new URLSearchParams()
    if (params.espai_type) queryParams.append('espai_type', params.espai_type)
    if (params.num_capacitat) queryParams.append('num_capacitat', params.num_capacitat)
    if (params.caracteristicas) {
      params.caracteristicas.forEach(caracteristica => {
        queryParams.append('caracteristicas', caracteristica)
      })
    }
    if (params.skip) queryParams.append('skip', params.skip)
    if (params.limit) queryParams.append('limit', params.limit)

    return http.get(`/api/v1/espai/filter_me?${queryParams.toString()}`, {
      headers: {
        'Authorization': `Bearer ${token}`
      }
    })
      .then(res => {
        return res.data
      })
      .catch(error => {
        console.error('Error filtering my espai', error)
        throw error
      })
  }
}

export default new EspaiService()
