import http from '@/http-common'

class FotografiaService {
  // Obtener todos los servicios de fotografía
  getAllFotografias (skip = 0, limit = 10) {
    return http.get(`/api/v1/fotografias/all/?skip=${skip}&limit=${limit}`)
      .then(res => {
        return res.data
      })
      .catch(error => {
        console.error('Error fetching fotografias', error)
        throw error
      })
  }

  // Obtener mis servicios de fotografía
  getAllMyFotografias (skip = 0, limit = 10) {
    const token = localStorage.getItem('token')
    return http.get(`/api/v1/fotografias/all_me/?skip=${skip}&limit=${limit}`, {
      headers: {
        'Authorization': `Bearer ${token}`
      }
    })
      .then(res => {
        return res.data
      })
      .catch(error => {
        console.error('Error fetching my fotografias', error)
        throw error
      })
  }

  // Obtener un servicio de fotografía por ID
  getFotografiaById (id) {
    return http.get(`/api/v1/fotografias/by_id/${id}`)
      .then(res => {
        return res.data
      })
      .catch(error => {
        console.error('Error fetching fotografia', error)
        throw error
      })
  }

  // Crear un nuevo servicio de fotografía
  createFotografia (fotografiaData) {
    const token = localStorage.getItem('token')
    return http.post('/api/v1/fotografias/', fotografiaData, {
      headers: {
        'Authorization': `Bearer ${token}`
      }
    })
      .then(res => {
        return res.data
      })
      .catch(error => {
        console.error('Error creating fotografia', error)
        throw error
      })
  }

  // Actualizar un servicio de fotografía
  updateFotografia (id, fotografiaData) {
    const token = localStorage.getItem('token')
    return http.patch(`/api/v1/fotografias/${id}`, fotografiaData, {
      headers: {
        'Authorization': `Bearer ${token}`
      }
    })
      .then(res => {
        return res.data
      })
      .catch(error => {
        console.error('Error updating fotografia', error)
        throw error
      })
  }

  // Actualizar parcialmente un servicio de fotografía
  patchFotografia (id, fotografiaData) {
    const token = localStorage.getItem('token')
    return http.patch(`/api/v1/fotografias/${id}`, fotografiaData, {
      headers: {
        'Authorization': `Bearer ${token}`
      }
    })
      .then(res => {
        return res.data
      })
      .catch(error => {
        console.error('Error patching fotografia', error)
        throw error
      })
  }

  // Eliminar un servicio de fotografía
  deleteFotografia (id) {
    const token = localStorage.getItem('token')
    return http.delete(`/api/v1/fotografias/${id}`, {
      headers: {
        'Authorization': `Bearer ${token}`
      }
    })
      .then(res => {
        return res.data
      })
      .catch(error => {
        console.error('Error deleting fotografia', error)
        throw error
      })
  }

  // Filtrar servicios de fotografía
  filterFotografias (params) {
    const queryParams = new URLSearchParams()
    if (params.fotografia_type) queryParams.append('fotografia_type', params.fotografia_type)
    if (params.servicio_especial) queryParams.append('servicio_especial', params.servicio_especial)
    if (params.min_duracion) queryParams.append('min_duracion', params.min_duracion)
    if (params.max_duracion) queryParams.append('max_duracion', params.max_duracion)
    if (params.skip) queryParams.append('skip', params.skip)
    if (params.limit) queryParams.append('limit', params.limit)

    return http.get(`/api/v1/fotografias/filter?${queryParams.toString()}`)
      .then(res => {
        return res.data
      })
      .catch(error => {
        console.error('Error filtering fotografias', error)
        throw error
      })
  }

  // Filtrar mis servicios de fotografía
  filterMyFotografias (params) {
    const token = localStorage.getItem('token')
    const queryParams = new URLSearchParams()
    if (params.fotografia_type) queryParams.append('fotografia_type', params.fotografia_type)
    if (params.servicio_especial) queryParams.append('servicio_especial', params.servicio_especial)
    if (params.min_duracion) queryParams.append('min_duracion', params.min_duracion)
    if (params.max_duracion) queryParams.append('max_duracion', params.max_duracion)
    if (params.skip) queryParams.append('skip', params.skip)
    if (params.limit) queryParams.append('limit', params.limit)

    return http.get(`/api/v1/fotografias/filter_me?${queryParams.toString()}`, {
      headers: {
        'Authorization': `Bearer ${token}`
      }
    })
      .then(res => {
        return res.data
      })
      .catch(error => {
        console.error('Error filtering my fotografias', error)
        throw error
      })
  }
}

export default new FotografiaService()
