# web/scripts/ens_controls.py

import pandas as pd
import os

class ENSControlLoader:
    """
    Carga todos los controles ENS Nivel Alto desde el Excel
    y genera, para cada fila, un diccionario con:
      - codigo
      - nombre
      - descripcion
      - pregunta: texto a mostrar en el formulario
    """

    def __init__(self, excel_path=None):
        """
        Si no se pasa excel_path, busca por defecto en 'static/data/ENS2022_nivel_ALTO.xlsx'.
        """
        if excel_path is None:
            base_dir = os.path.dirname(os.path.abspath(__file__))
            default_path = os.path.join(base_dir, os.pardir, 'static', 'data', 'ENS2022_nivel_ALTO.xlsx')
            excel_path = os.path.abspath(default_path)

        if not os.path.isfile(excel_path):
            raise FileNotFoundError(f"No encontré el fichero de controles ENS en: {excel_path}")

        df = pd.read_excel(excel_path, dtype=str)

        expected_cols = {'Código ENS', 'Nombre del Control', 'Explicación Breve'}
        if not expected_cols.issubset(set(df.columns)):
            raise ValueError(f"El Excel debe contener columnas {expected_cols}, "
                             f"pero encontré: {df.columns.tolist()}")

        # Eliminamos filas sin código
        df = df.dropna(subset=['Código ENS'])

        # Normalizamos texto y creamos la lista de controles
        self.controles = []
        for _, row in df.iterrows():
            codigo = str(row['Código ENS']).strip()
            nombre = str(row['Nombre del Control']).strip()
            descripcion = str(row['Explicación Breve']).strip()

            pregunta = (
                f"¿Dispone la organización de: {descripcion}? "
                f"(Control {codigo} – {nombre})"
            )

            self.controles.append({
                'codigo': codigo,
                'nombre': nombre,
                'descripcion': descripcion,
                'pregunta': pregunta
            })

    def load_questions(self):
        """
        Devuelve la lista de controles en formato de diccionario:
          [
            { 'codigo': 'OP.ACC.1',
              'nombre': 'Identificación',
              'descripcion': 'Describiendo brevemente qué exige este control …',
              'pregunta': '¿Dispone la organización de: ...? (Control OP.ACC.1 – Identificación)'
            },
            ...
          ]
        """
        return self.controles
