import pandas as pd

# ═══════════════════════════════════════════════════════════════════════════════
# RECOMENDACIONES ESPECÍFICAS POR CONTROL ENS
# ═══════════════════════════════════════════════════════════════════════════════

# Estructura: "CODIGO_ENS": {
#   "cumple": "Mensaje cuando cumple",
#   "parcial": "Mensaje cuando cumple parcialmente", 
#   "no_cumple": "Recomendación específica cuando no cumple"
# }

RECOMENDACIONES_ESPECIFICAS = {
    # ══════════════════════════════════════════════════════════════════════════
    # MEDIDAS DE PROTECCIÓN (MP.*) 
    # ══════════════════════════════════════════════════════════════════════════
    
    # MP.COM.* - Protección de las comunicaciones
    "MP.COM.1": {
        "cumple": "Excelente. La integridad de las comunicaciones está garantizada.",
        "parcial": "Mejora necesaria: Refuerce los mecanismos de integridad de comunicaciones. Considere implementar certificados digitales adicionales.",
        "no_cumple": "CRÍTICO: Implemente protección de integridad en comunicaciones mediante certificados digitales, firmas electrónicas o HTTPS. Ejecute el Script 6 (Simulador) para identificar vulnerabilidades de comunicación."
    },
    "MP.COM.2": {
        "cumple": "Correcto. La confidencialidad en las comunicaciones está protegida.",
        "parcial": "Mejora recomendada: Refuerce el cifrado en todas las comunicaciones críticas. Revise configuraciones TLS/SSL.",
        "no_cumple": "CRÍTICO: Implemente cifrado en comunicaciones (TLS 1.2+, VPN, túneles seguros). Use el Script 6 para detectar servicios no cifrados expuestos."
    },
    "MP.COM.3": {
        "cumple": "Perfecto. La autenticidad de las comunicaciones está verificada.",
        "parcial": "Acción necesaria: Complete la implementación de autenticación mutua en todas las comunicaciones críticas.",
        "no_cumple": "CRÍTICO: Implemente autenticación de extremo a extremo (certificados mutuos, PKI). El Script 3 puede ayudar a generar políticas de comunicación segura."
    },
    "MP.COM.4": {
        "cumple": "Bien. La segregación de redes está implementada correctamente.",
        "parcial": "Mejora recomendada: Complete la segregación de redes críticas. Revise VLANs y subredes.",
        "no_cumple": "ALTO: Implemente segregación de redes mediante VLANs, subredes o zonas DMZ. Use el Script 7 para estimar costos de equipamiento de red."
    },

    # MP.EQ.* - Protección de equipos
    "MP.EQ.1": {
        "cumple": "Correcto. Los equipos están protegidos en el puesto de trabajo.",
        "parcial": "Mejora recomendada: Refuerce la protección física de equipos adicionales (portátiles, dispositivos móviles).",
        "no_cumple": "MEDIO: Implemente protección física de equipos (anclajes, alarmas, control de acceso). El Script 7 puede estimar costos de medidas físicas."
    },
    "MP.EQ.2": {
        "cumple": "Bien. Los equipos portátiles están protegidos adecuadamente.",
        "parcial": "Acción necesaria: Complete el cifrado de discos y políticas de uso de equipos portátiles.",
        "no_cumple": "ALTO: Implemente cifrado de discos (BitLocker/LUKS), políticas de uso y control remoto. Use el Script 4 para auditar configuraciones de seguridad."
    },
    "MP.EQ.3": {
        "cumple": "Excelente. La protección de servidores está implementada.",
        "parcial": "Mejora recomendada: Refuerce la protección física y lógica de servidores adicionales.",
        "no_cumple": "CRÍTICO: Implemente protección de servidores (acceso físico restringido, redundancia, monitorización). El Script 5 puede detectar amenazas a servidores."
    },
    "MP.EQ.4": {
        "cumple": "Correcto. Los dispositivos de almacenamiento están protegidos.",
        "parcial": "Acción necesaria: Complete el cifrado y control de acceso a dispositivos de almacenamiento.",
        "no_cumple": "ALTO: Implemente cifrado de almacenamiento, control de dispositivos USB y políticas de almacenamiento. Use el Script 3 para generar políticas específicas."
    },

    # MP.IF.* - Protección de instalaciones y servicios
    "MP.IF.1": {
        "cumple": "Bien. Las áreas de trabajo están adecuadamente protegidas.",
        "parcial": "Mejora recomendada: Refuerce el control de acceso a áreas sensibles adicionales.",
        "no_cumple": "MEDIO: Implemente control de acceso físico (tarjetas, biometría, cámaras). El Script 7 puede estimar costos de sistemas de seguridad física."
    },
    "MP.IF.2": {
        "cumple": "Correcto. La protección física de instalaciones es adecuada.",
        "parcial": "Acción necesaria: Complete las medidas de protección perimetral y sistemas de alarma.",
        "no_cumple": "ALTO: Implemente protección perimetral (vallado, iluminación, CCTV, sistemas de alarma). Consulte el Script 7 para planificación de recursos."
    },
    "MP.IF.3": {
        "cumple": "Excelente. El acondicionamiento e instalaciones están protegidos.",
        "parcial": "Mejora recomendada: Refuerce sistemas de climatización y protección eléctrica en áreas críticas.",
        "no_cumple": "MEDIO: Implemente sistemas de climatización, SAI, protección contra incendios. Use el Script 7 para estimar inversión en infraestructura."
    },
    "MP.IF.4": {
        "cumple": "Bien. El mantenimiento de instalaciones está controlado.",
        "parcial": "Acción necesaria: Complete los procedimientos de mantenimiento y supervisión de personal externo.",
        "no_cumple": "MEDIO: Establezca procedimientos de mantenimiento seguro y supervisión de contratistas. El Script 3 puede generar estas políticas."
    },
    "MP.IF.5": {
        "cumple": "Correcto. La protección contra amenazas ambientales está implementada.",
        "parcial": "Mejora recomendada: Refuerce las medidas contra amenazas ambientales específicas de su ubicación.",
        "no_cumple": "MEDIO: Implemente protección contra agua, fuego, terremotos según riesgos locales. Consulte el Script 7 para planificación."
    },
    "MP.IF.6": {
        "cumple": "Bien. El suministro de servicios está protegido adecuadamente.",
        "parcial": "Acción necesaria: Complete la redundancia en servicios críticos (electricidad, comunicaciones).",
        "no_cumple": "ALTO: Implemente redundancia en suministros críticos (doble acometida eléctrica, ISP backup). Use el Script 7 para análisis de costos."
    },
    "MP.IF.7": {
        "cumple": "Excelente. El cableado está protegido correctamente.",
        "parcial": "Mejora recomendada: Complete la protección de cableado en áreas sensibles adicionales.",
        "no_cumple": "MEDIO: Proteja el cableado mediante canaletas, armarios cerrados y rutas no accesibles. Consulte el Script 7 para estimación."
    },

    # MP.INFO.* - Protección de la información
    "MP.INFO.1": {
        "cumple": "Correcto. Los datos están clasificados adecuadamente.",
        "parcial": "Acción necesaria: Complete la clasificación de información según niveles de sensibilidad.",
        "no_cumple": "ALTO: Implemente un esquema de clasificación de datos (público, interno, confidencial, restringido). Use el Script 3 para generar políticas de clasificación."
    },
    "MP.INFO.2": {
        "cumple": "Bien. El etiquetado de información está implementado.",
        "parcial": "Mejora recomendada: Complete el etiquetado automático y manual de documentos sensibles.",
        "no_cumple": "MEDIO: Implemente etiquetado de documentos y correos según clasificación. El Script 3 puede generar plantillas y políticas."
    },
    "MP.INFO.4": {
        "cumple": "Excelente. El almacenamiento de información está protegido.",
        "parcial": "Mejora recomendada: Refuerce el cifrado y control de acceso a almacenamiento crítico.",
        "no_cumple": "CRÍTICO: Implemente cifrado de almacenamiento, backups seguros y control de acceso estricto. Use el Script 4 para auditar permisos de archivos."
    },
    "MP.INFO.5": {
        "cumple": "Correcto. Las copias de seguridad están protegidas adecuadamente.",
        "parcial": "Acción necesaria: Complete la estrategia de backups y pruebas de restauración regulares.",
        "no_cumple": "CRÍTICO: Implemente copias de seguridad automáticas, cifradas y con pruebas de restauración. Use el Script 7 para planificar infraestructura de backup."
    },
    "MP.INFO.6": {
        "cumple": "Bien. La destrucción de información está controlada.",
        "parcial": "Mejora recomendada: Complete los procedimientos seguros de borrado en todos los dispositivos.",
        "no_cumple": "MEDIO: Implemente borrado seguro de datos (software especializado, destrucción física). El Script 3 puede generar procedimientos específicos."
    },
    "MP.INFO.9": {
        "cumple": "Excelente. La protección de información fuera de instalaciones está implementada.",
        "parcial": "Mejora recomendada: Refuerce las medidas de protección para trabajo remoto y dispositivos móviles.",
        "no_cumple": "ALTO: Implemente VPN, cifrado de dispositivos móviles y políticas de trabajo remoto. Use el Script 3 para generar políticas de teletrabajo."
    },

    # MP.PER.* - Protección de personal
    "MP.PER.1": {
        "cumple": "Correcto. Las condiciones de personal están controladas.",
        "parcial": "Acción necesaria: Complete los procedimientos de verificación de antecedentes según el puesto.",
        "no_cumple": "MEDIO: Implemente verificación de antecedentes, claúsulas de confidencialidad y procedimientos de contratación segura. Use el Script 3 para generar plantillas."
    },
    "MP.PER.2": {
        "cumple": "Bien. Las obligaciones del personal están definidas.",
        "parcial": "Mejora recomendada: Refuerce la formación específica en ciberseguridad según el rol.",
        "no_cumple": "MEDIO: Establezca obligaciones claras de seguridad, acuerdos de confidencialidad específicos. El Script 3 puede generar documentación de RRHH."
    },
    "MP.PER.3": {
        "cumple": "Excelente. La concienciación del personal está implementada.",
        "parcial": "Mejora recomendada: Amplíe los programas de concienciación y actualice contenidos regularmente.",
        "no_cumple": "ALTO: Implemente programa de concienciación en ciberseguridad (formación regular, simulacros de phishing). Use el Script 7 para planificar formación."
    },
    "MP.PER.4": {
        "cumple": "Correcto. La terminación o cambio de puesto están controlados.",
        "parcial": "Acción necesaria: Complete los procedimientos de revocación de accesos y entrega de equipos.",
        "no_cumple": "ALTO: Implemente procedimientos de offboarding (revocación inmediata de accesos, entrega de equipos, entrevista de salida). Use el Script 4 para auditar cuentas activas."
    },

    # MP.S.* - Protección de los servicios
    "MP.S.1": {
        "cumple": "Bien. La configuración de seguridad está controlada.",
        "parcial": "Mejora recomendada: Aplique configuraciones de seguridad ('hardening') en sistemas adicionales.",
        "no_cumple": "ALTO: Implemente configuraciones seguras (hardening) en todos los sistemas. Use el Script 4 para auditar configuraciones actuales."
    },
    "MP.S.2": {
        "cumple": "Correcto. La gestión de vulnerabilidades está implementada.",
        "parcial": "Acción necesaria: Establezca un programa regular de gestión de vulnerabilidades y parches.",
        "no_cumple": "CRÍTICO: Implemente gestión de vulnerabilidades (escaneo regular, gestión de parches, WSUS/SCCM). Use el Script 6 para identificar vulnerabilidades actuales."
    },
    "MP.S.3": {
        "cumple": "Excelente. La protección frente a código dañino está implementada.",
        "parcial": "Mejora recomendada: Refuerce la protección antimalware en todos los endpoints y servidores.",
        "no_cumple": "CRÍTICO: Implemente solución antimalware corporativa (antivirus centralizado, EDR, sandboxing). Use el Script 7 para estimar costos de licencias."
    },
    "MP.S.8": {
        "cumple": "Correcto. Las pruebas de seguridad están implementadas.",
        "parcial": "Acción necesaria: Establezca un programa regular de pentesting y auditorías de seguridad.",
        "no_cumple": "ALTO: Implemente pruebas de penetración regulares (internas/externas, auditorías de código). Use el Script 6 para simulaciones básicas y el Script 7 para planificar auditorías externas."
    },

    # ══════════════════════════════════════════════════════════════════════════
    # MEDIDAS OPERACIONALES (OP.*)
    # ══════════════════════════════════════════════════════════════════════════

    # OP.ACC.* - Control de acceso
    "OP.ACC.1": {
        "cumple": "Excelente. La identificación está correctamente implementada.",
        "parcial": "Mejora necesaria: Refuerce los mecanismos de identificación única para todos los usuarios.",
        "no_cumple": "CRÍTICO: Implemente identificación única para cada usuario (cuentas nominales, no compartidas). Ejecute el Script 4 para auditar cuentas duplicadas o compartidas."
    },
    "OP.ACC.2": {
        "cumple": "Correcto. Los requisitos de acceso están bien definidos.",
        "parcial": "Acción recomendada: Complete la definición de perfiles de acceso según roles organizacionales.",
        "no_cumple": "ALTO: Defina requisitos de acceso por roles y recursos. Implemente matriz de accesos. Use el Script 3 para generar políticas de control de acceso."
    },
    "OP.ACC.3": {
        "cumple": "Bien. La segregación de funciones está implementada.",
        "parcial": "Mejora recomendada: Refuerce la separación de funciones críticas (administración, operación, auditoría).",
        "no_cumple": "ALTO: Implemente segregación de funciones para evitar conflictos de interés. Separe roles de administración, operación y control. Use el Script 4 para identificar usuarios con privilegios excesivos."
    },
    "OP.ACC.4": {
        "cumple": "Excelente. El proceso de gestión de derechos de acceso funciona correctamente.",
        "parcial": "Acción necesaria: Mejore los procesos de revisión y actualización de derechos de acceso.",
        "no_cumple": "CRÍTICO: Implemente gestión de derechos con principio de menor privilegio, revisiones periódicas y workflow de aprobación. Use el Script 4 para identificar privilegios excesivos."
    },
    "OP.ACC.5": {
        "cumple": "Correcto. La autenticación de usuarios externos está bien configurada.",
        "parcial": "Mejora recomendada: Refuerce la autenticación multifactor para todos los accesos externos.",
        "no_cumple": "CRÍTICO: Implemente autenticación robusta para usuarios externos (MFA, VPN, certificados). Use el Script 4 para auditar accesos RDP y conexiones externas."
    },
    "OP.ACC.6": {
        "cumple": "Bien. La autenticación de usuarios internos está configurada.",
        "parcial": "Acción recomendada: Mejore las políticas de contraseñas y considere implementar MFA interno.",
        "no_cumple": "ALTO: Implemente políticas de contraseña robustas, MFA para administradores y autenticación de dominio. Use el Script 4 para auditar políticas de contraseña actuales."
    },

    # OP.CONT.* - Explotación
    "OP.CONT.1": {
        "cumple": "Excelente. Los procedimientos de operación están documentados.",
        "parcial": "Mejora recomendada: Complete y actualice procedimientos operacionales críticos faltantes.",
        "no_cumple": "ALTO: Documente procedimientos de operación, manuales de usuario y protocolos de emergencia. Use el Script 3 para generar documentación operacional."
    },
    "OP.CONT.2": {
        "cumple": "Correcto. La gestión de cambios está controlada.",
        "parcial": "Acción necesaria: Refuerce el control de cambios en sistemas críticos y documentación.",
        "no_cumple": "ALTO: Implemente gestión formal de cambios (comité de cambios, pruebas, rollback). Use el Script 3 para generar procedimientos de gestión de cambios."
    },
    "OP.CONT.3": {
        "cumple": "Bien. La gestión de la configuración está implementada.",
        "parcial": "Mejora recomendada: Complete el inventario y control de configuraciones de todos los sistemas.",
        "no_cumple": "MEDIO: Implemente gestión de configuración (CMDB, versionado, líneas base). Use herramientas como Ansible o PowerShell DSC."
    },
    "OP.CONT.4": {
        "cumple": "Correcto. El mantenimiento está controlado adecuadamente.",
        "parcial": "Acción necesaria: Establezca ventanas de mantenimiento y procedimientos de supervisión más estrictos.",
        "no_cumple": "MEDIO: Implemente mantenimiento controlado (ventanas programadas, supervisión, documentación). Use el Script 3 para generar políticas de mantenimiento."
    },

    # OP.EXP.* - Explotación
    "OP.EXP.1": {
        "cumple": "Excelente. La explotación del sistema está bien gestionada.",
        "parcial": "Mejora recomendada: Refuerce los procedimientos de operación y monitorización del sistema.",
        "no_cumple": "MEDIO: Establezca procedimientos de operación diaria, monitorización proactiva y mantenimiento preventivo."
    },
    "OP.EXP.2": {
        "cumple": "Correcto. La gestión de la integridad está implementada.",
        "parcial": "Acción necesaria: Refuerce los controles de integridad de datos y sistemas críticos.",
        "no_cumple": "ALTO: Implemente controles de integridad (checksums, firmas digitales, monitorización de cambios). Use herramientas como AIDE o Tripwire."
    },
    "OP.EXP.3": {
        "cumple": "Bien. La gestión de incidentes está funcionando correctamente.",
        "parcial": "Mejora recomendada: Refuerce los procedimientos de respuesta y recuperación ante incidentes.",
        "no_cumple": "CRÍTICO: Implemente gestión formal de incidentes (CSIRT, procedimientos, herramientas). Use el Script 5 para detectar incidentes y el Script 3 para generar procedimientos."
    },
    "OP.EXP.4": {
        "cumple": "Excelente. La protección frente a código malicioso está implementada.",
        "parcial": "Acción necesaria: Refuerce la protección antimalware y actualice definiciones más frecuentemente.",
        "no_cumple": "CRÍTICO: Implemente protección integral contra malware (antivirus, EDR, sandboxing, filtrado web). Use el Script 7 para planificar soluciones."
    },
    "OP.EXP.5": {
        "cumple": "Correcto. La gestión de vulnerabilidades funciona adecuadamente.",
        "parcial": "Mejora recomendada: Acelere los tiempos de respuesta y cobertura del programa de vulnerabilidades.",
        "no_cumple": "CRÍTICO: Implemente gestión de vulnerabilidades (escáner automático, gestión de parches, evaluación de riesgo). Use el Script 6 para identificar vulnerabilidades."
    },
    "OP.EXP.6": {
        "cumple": "Bien. La gestión de logs está implementada.",
        "parcial": "Acción necesaria: Mejore la centralización y análisis automatizado de logs.",
        "no_cumple": "ALTO: Implemente gestión centralizada de logs (SIEM, syslog, retención). Use el Script 5 para detectar eventos de seguridad y el Script 7 para estimar costos de SIEM."
    },
    "OP.EXP.7": {
        "cumple": "Excelente. La monitorización del sistema está funcionando.",
        "parcial": "Mejora recomendada: Amplíe la cobertura de monitorización y mejore la respuesta automatizada.",
        "no_cumple": "ALTO: Implemente monitorización integral (rendimiento, seguridad, disponibilidad). Use el Script 5 para monitorización de incidentes."
    },
    "OP.EXP.8": {
        "cumple": "Correcto. La gestión de la capacidad está controlada.",
        "parcial": "Acción necesaria: Mejore la planificación de capacidad y alertas proactivas.",
        "no_cumple": "MEDIO: Implemente gestión de capacidad (monitorización de recursos, planificación, alertas). Use herramientas de monitorización como Nagios o Zabbix."
    },
    "OP.EXP.9": {
        "cumple": "Bien. Las copias de seguridad están gestionadas correctamente.",
        "parcial": "Mejora recomendada: Refuerce las pruebas de restauración y almacenamiento offsite.",
        "no_cumple": "CRÍTICO: Implemente estrategia integral de backups (automatización, cifrado, pruebas, offsite). Use el Script 7 para planificar infraestructura de backup."
    },

    # OP.EXT.* - Externalización
    "OP.EXT.1": {
        "cumple": "Excelente. La externalización está bien controlada.",
        "parcial": "Mejora recomendada: Refuerce los controles sobre proveedores críticos y cláusulas de seguridad.",
        "no_cumple": "ALTO: Implemente gestión de riesgos de terceros (evaluación, contratos de seguridad, auditorías). Use el Script 3 para generar cláusulas contractuales."
    },
    "OP.EXT.2": {
        "cumple": "Correcto. Los acuerdos de nivel de servicio están definidos.",
        "parcial": "Acción necesaria: Refuerce los SLA de seguridad y métricas de cumplimiento.",
        "no_cumple": "MEDIO: Defina SLA específicos de seguridad con proveedores (tiempo de respuesta, disponibilidad, incidentes). Use el Script 3 para plantillas de SLA."
    },
    "OP.EXT.3": {
        "cumple": "Bien. La gestión diaria de la externalización funciona correctamente.",
        "parcial": "Mejora recomendada: Refuerce la supervisión continua y reporting de proveedores.",
        "no_cumple": "MEDIO: Implemente supervisión continua de servicios externalizados (KPI, reportes, reuniones). Establezca procedimientos de escalado."
    },
    "OP.EXT.4": {
        "cumple": "Excelente. La terminación de la externalización está controlada.",
        "parcial": "Acción necesaria: Complete los procedimientos de terminación y recuperación de datos.",
        "no_cumple": "MEDIO: Defina procedimientos de terminación segura (recuperación de datos, destrucción, transición). Use el Script 3 para documentar procesos."
    },

    # OP.NUB.* - Servicios en la nube
    "OP.NUB.1": {
        "cumple": "Correcto. El uso de servicios en la nube está controlado.",
        "parcial": "Mejora recomendada: Refuerce los controles de seguridad y compliance en servicios cloud.",
        "no_cumple": "ALTO: Implemente gestión de servicios cloud (política de uso, evaluación de riesgos, configuración segura). Use el Script 3 para políticas cloud."
    },

    # OP.PL.* - Planificación
    "OP.PL.1": {
        "cumple": "Bien. La planificación está implementada correctamente.",
        "parcial": "Mejora recomendada: Refuerce la planificación estratégica de seguridad y revisiones periódicas.",
        "no_cumple": "ALTO: Implemente planificación formal de seguridad (estrategia, roadmap, presupuesto). Use el Script 7 para optimización de recursos y planificación."
    },
    "OP.PL.2": {
        "cumple": "Excelente. La arquitectura de seguridad está bien definida.",
        "parcial": "Acción necesaria: Complete la documentación de arquitectura y actualice según cambios.",
        "no_cumple": "ALTO: Defina arquitectura de seguridad formal (zonificación, flujos de datos, controles). Use el Script 3 para documentar arquitectura."
    },
    "OP.PL.3": {
        "cumple": "Correcto. La adquisición de nuevos componentes está controlada.",
        "parcial": "Mejora recomendada: Refuerce la evaluación de seguridad en procesos de adquisición.",
        "no_cumple": "MEDIO: Implemente evaluación de seguridad en adquisiciones (criterios, homologación, pruebas). Use el Script 7 para análisis de costos vs seguridad."
    },
    "OP.PL.4": {
        "cumple": "Bien. Los componentes certificados están siendo utilizados.",
        "parcial": "Acción necesaria: Priorice el uso de componentes certificados en sistemas críticos.",
        "no_cumple": "MEDIO: Use productos certificados (Common Criteria, FIPS 140-2) especialmente para funciones criptográficas críticas."
    },
    "OP.PL.5": {
        "cumple": "Excelente. Las pruebas de seguridad están implementadas.",
        "parcial": "Mejora recomendada: Amplíe la cobertura y frecuencia de las pruebas de seguridad.",
        "no_cumple": "ALTO: Implemente programa de pruebas de seguridad (pentesting, análisis de código, auditorías). Use el Script 6 para pruebas básicas."
    },

    # ══════════════════════════════════════════════════════════════════════════
    # ORGANIZACIÓN (ORG.*)
    # ══════════════════════════════════════════════════════════════════════════
    
    "ORG.1": {
        "cumple": "Excelente. La política de seguridad está bien establecida.",
        "parcial": "Mejora recomendada: Actualice y refine la política de seguridad según cambios organizacionales.",
        "no_cumple": "CRÍTICO: Desarrolle e implemente política de seguridad formal aprobada por dirección. Use el Script 3 para generar políticas RGPD/ENS completas."
    },
    "ORG.2": {
        "cumple": "Correcto. Los procedimientos de seguridad están documentados.",
        "parcial": "Acción necesaria: Complete la documentación de procedimientos críticos faltantes.",
        "no_cumple": "ALTO: Documente procedimientos de seguridad detallados (manuales, protocolos, checklists). Use el Script 3 para generar documentación completa."
    },
    "ORG.3": {
        "cumple": "Bien. El proceso de autorización está funcionando.",
        "parcial": "Mejora recomendada: Refuerce los procesos de autorización y documentación de decisiones.",
        "no_cumple": "ALTO: Implemente proceso formal de autorización de sistemas (acreditación, documentación, revisiones). Designe autoridades de acreditación."
    },
    "ORG.4": {
        "cumple": "Excelente. La función de seguridad está establecida.",
        "parcial": "Acción necesaria: Refuerce las competencias y recursos del responsable de seguridad.",
        "no_cumple": "CRÍTICO: Designe Oficial de Seguridad con competencias y autoridad adecuadas. Use el Script 7 para estimar costos de recursos de seguridad."
    }
}

def generate_diagnostic(respuestas):
    """
    A partir de la lista de respuestas generada por EvaluadorENS (cada elemento con
    'codigo', 'respuesta' y, en caso de 'parcial', 'porcentaje'), construye un
    DataFrame con las columnas:
      - 'Código ENS'       (p. ej. "OP.ACC.5")
      - 'Estado'           ("Cumple", "Parcial (XX%)" o "No cumple")
      - 'Recomendación'    (recomendación específica según el estado)
    
    Devuelve un pandas.DataFrame ordenado por 'Código ENS'.
    """

    filas = []
    for item in respuestas:
        codigo = item.get('codigo')
        resp = item.get('respuesta')
        pct = item.get('porcentaje')

        # (1) Determinar el campo "Estado"
        if resp == "cumple":
            estado = "Cumple"
        elif resp == "parcial" and isinstance(pct, int):
            estado = f"Parcial ({pct}%)"
        else:
            estado = "No cumple"

        # (2) Buscar recomendación específica según el estado
        if codigo in RECOMENDACIONES_ESPECIFICAS:
            recomendaciones = RECOMENDACIONES_ESPECIFICAS[codigo]
            
            if resp == "cumple":
                reco = recomendaciones.get("cumple", "Control implementado correctamente.")
            elif resp == "parcial":
                reco = recomendaciones.get("parcial", "Mejoras necesarias para alcanzar cumplimiento completo.")
            else:  # no cumple
                reco = recomendaciones.get("no_cumple", "Implementación urgente requerida.")
        else:
            # Fallback para controles no definidos
            if resp == "cumple":
                reco = "✅ Control implementado correctamente. Mantenga las medidas actuales."
            elif resp == "parcial":
                reco = "⚠️ Mejoras necesarias: Refuerce las medidas existentes para alcanzar cumplimiento completo."
            else:
                reco = "🚨 Implementación urgente requerida: Este control debe ser implementado para cumplir con ENS."

        filas.append({
            "Código ENS": codigo,
            "Estado": estado,
            "Recomendación": reco
        })

    # Construir el DataFrame y ordenarlo por 'Código ENS'
    df = pd.DataFrame(filas)
    df = df.sort_values(by="Código ENS").reset_index(drop=True)
    return df
