#!/usr/bin/env bash
# Adapted training script for fine-tuning only the tiny (small) model on a single Google Cloud GPU.

# Use a single GPU: Google Cloud GPU or Colab GPU (e.g., Tesla T4)
CUDA_VISIBLE_DEVICES=0

# Set paths for modules and utilities
user_dir=../../module
bpe_dir=../../utils/BPE

# Set the data directory and training/validation files (TSV format)
dataset_name=$1
data_dir=../../datasets/finetuning/${dataset_name}
data=${data_dir}/train.tsv,${data_dir}/val.tsv
ans2label_file=${data_dir}/trainval_ans2label.pkl

# For testing, we only use the tiny model.
scale="medium"

# Set image patch size and batch size based on the model scale.
if [[ $scale =~ "medium" ]]; then
    batch_size=32
    patch_image_size=256
fi

# Set the pre-trained checkpoint for the tiny model.
restore_file=../../checkpoints/biomedgpt_${scale}.pt
selected_cols=0,5,2,3,4

# Define directories for logs and saving checkpoints.
log_dir=./${dataset_name}_logs/${scale}
save_dir=../../checkpoints/tuned_checkpoints/${dataset_name}/${scale}
mkdir -p $log_dir $save_dir

# Define task and architecture details.
task=vqa_gen
arch=ofa_${scale}
criterion=adjust_label_smoothed_cross_entropy
label_smoothing=0.05

# Update frequency for gradient accumulation.
update_freq=4

# Dropout and regularization parameters.
resnet_drop_path_rate=0.0
encoder_drop_path_rate=0.1
decoder_drop_path_rate=0.1
dropout=0.1
attention_dropout=0.0

# Maximum sequence lengths.
max_src_length=80
max_object_length=30
max_tgt_length=40
num_bins=1000

# Exponential Moving Average (EMA) configuration.
uses_ema="--uses-ema"
store_ema="--store-ema"
ema_fp32="--ema-fp32"
ema_decay=0.9999
ema_start_update=0

# Set validation inference type.
val_inference_type=beamsearch

# For testing, we disable any unconstrained training flags.
unconstrained_training_flag=""

# Set a single hyperparameter configuration for testing.
max_epoch=15
warmup_ratio=0.04
lr=5e-5

# Create unique log file and save directory for the current configuration.
log_file=${log_dir}/${max_epoch}"_"${warmup_ratio}"_"${lr}"_"${patch_image_size}".log"
save_path=${save_dir}/${max_epoch}"_"${warmup_ratio}"_"${lr}"_"${patch_image_size}
mkdir -p $save_path

# Run training on a single GPU.
python3 ../../train.py \
    ${data} \
    --selected-cols=${selected_cols} \
    --bpe-dir=${bpe_dir} \
    --user-dir=${user_dir} \
    --restore-file=${restore_file} \
    --reset-optimizer --reset-dataloader --reset-meters \
    --save-dir=${save_path} \
    --task=${task} \
    --arch=${arch} \
    --criterion=${criterion} \
    --label-smoothing=${label_smoothing} \
    --batch-size=${batch_size} \
    --update-freq=${update_freq} \
    --encoder-normalize-before \
    --decoder-normalize-before \
    --share-decoder-input-output-embed \
    --share-all-embeddings \
    --layernorm-embedding \
    --patch-layernorm-embedding \
    --code-layernorm-embedding \
    --resnet-drop-path-rate=${resnet_drop_path_rate} \
    --encoder-drop-path-rate=${encoder_drop_path_rate} \
    --decoder-drop-path-rate=${decoder_drop_path_rate} \
    --dropout=${dropout} \
    --attention-dropout=${attention_dropout} \
    --weight-decay=0.01 \
    --optimizer=adam \
    --adam-betas="(0.9,0.999)" \
    --adam-eps=1e-08 \
    --clip-norm=1.0 \
    --lr-scheduler=polynomial_decay \
    --lr=${lr} \
    --max-epoch=${max_epoch} \
    --warmup-ratio=${warmup_ratio} \
    --log-format=simple \
    --log-interval=10 \
    --fixed-validation-seed=7 \
    --keep-last-epochs=1 \
    --save-interval=1 --validate-interval=1 \
    --best-checkpoint-metric=vqa_score --maximize-best-checkpoint-metric \
    --max-src-length=${max_src_length} \
    --max-object-length=${max_object_length} \
    --max-tgt-length=${max_tgt_length} \
    --find-unused-parameters \
    ${unconstrained_training_flag} \
    --ans2label-file=${ans2label_file} \
    --valid-batch-size=20 \
    --add-type-embedding \
    --scale-attn \
    --scale-fc \
    --scale-heads \
    --disable-entangle \
    --num-bins=${num_bins} \
    --patch-image-size=${patch_image_size} \
    --prompt-type=prev_output \
    --fp16 \
    --fp16-scale-window=512 \
    --add-object \
    ${uses_ema} \
    ${store_ema} \
    ${ema_fp32} \
    --ema-decay=${ema_decay} \
    --ema-start-update=${ema_start_update} \
    --val-inference-type=${val_inference_type} \
    --num-workers=0 \
    > ${log_file} 2>&1
