using UnityEngine;
using UnityEngine.UI;
using TMPro;
using System.Collections;

public class Warning : MonoBehaviour
{
    [SerializeField] private Image icon;
    [SerializeField] private Image background;
    [SerializeField] private Sprite[] sprites;
    [SerializeField] private Color positiveColor;
    [SerializeField] private Color warningColor;
    [SerializeField] private Color negativeColor;
    [SerializeField] private TextMeshProUGUI warningText;
    [SerializeField] private Animator animator;

    private Coroutine closeWarningCoroutine; // Store the coroutine reference

    public enum WarningType
    {
        Positive,
        Warning,
        Negative
    }

    // Show warning with default icon (Warning)
    public void ShowWarning(string warning)
    {
        ShowWarning(warning, WarningType.Warning); // Default to Warning icon type
    }

    // Show success with a specified icon type
    public void ShowSuccess(string warning)
    {
        ShowWarning(warning, WarningType.Positive); // Use Positive icon type
    }

    // Show error with a specified icon type
    public void ShowError(string warning)
    {
        ShowWarning(warning, WarningType.Negative); // Use Negative icon type
    }

    // Show warning with a specified icon type
    public void ShowWarning(string warning, WarningType warningType)
    {
        // Stop the existing coroutine if it's running
        if (closeWarningCoroutine != null)
        {
            StopCoroutine(closeWarningCoroutine);
        }

        gameObject.SetActive(true);
        animator.SetTrigger("Show");
        warningText.text = warning;

        // Set the icon and background based on the warning type
        SetWarningAppearance(warningType);

        closeWarningCoroutine = StartCoroutine(CloseWarning(3f)); // Start new coroutine and store its reference
    }

    // Set icon and background color based on warning type
    private void SetWarningAppearance(WarningType warningType)
    {
        switch (warningType)
        {

            case WarningType.Negative:
                icon.sprite = sprites[0]; // Icon for negative
                background.color = negativeColor;
                break;
            case WarningType.Warning:
                icon.sprite = sprites[1]; // Default icon
                background.color = warningColor;
                break;
            case WarningType.Positive:
                icon.sprite = sprites[2]; // Icon for positive
                background.color = positiveColor;
                break;
        }
    }

    private IEnumerator CloseWarning(float duration)
    {
        yield return new WaitForSecondsRealtime(duration);
        HideWarning();
    }

    public void HideWarning()
    {
        gameObject.SetActive(false);
    }
}
