using UnityEngine;
using TMPro; // Import TextMeshPro namespace
using System.Collections;

public class TimeController : MonoBehaviour
{
    public bool isPaused = false; // Tracks if the game is paused
    public bool gameStarted = false; // Tracks if the game has started
    public bool gameEnded = false; // Tracks if the game has ended
    public float elapsedTime = 0f; // Tracks elapsed time in seconds
    [SerializeField] private bool isEditorMode = false; // Tracks if the game is in editor mode
    public TextMeshProUGUI timerText; // Reference to a TextMeshProUGUI for displaying the timer

    // TODO update gameStarted when the "Par Champion" menu is added
    private void Awake()
    {
        bool isVerifyMode = PlayerPrefs.GetInt("isVerifyMode", 0) == 1; // Check if the game is in verify mode
        
        if (gameStarted) return; // Do not initialize the timer if the game has already started

        if (isEditorMode && !isVerifyMode)
        {
            gameStarted = true; // Set the game started state
            PlayerPrefs.SetInt("isPaused", 0); // Save the paused state
            Time.timeScale = 1f; // Resume the game
        } else
        {
            PlayerPrefs.SetInt("isPaused", 1); // Save the paused state
            Time.timeScale = 0f; // Stop the game by default. 
        }
        Debug.Log("Init Time scale: " + Time.timeScale);
    }
    private void Start()
    {
        //StartCoroutine(Initialize());
    }

    private IEnumerator Initialize()
    {
        yield return new WaitForEndOfFrame(); // Wait for the end of the frame to ensure all UI elements are initialized

        bool isVerifyMode = PlayerPrefs.GetInt("isVerifyMode", 0) == 1; // Check if the game is in verify mode
        
        if (gameStarted) yield break; // Do not initialize the timer if the game has already started

        if (isEditorMode && !isVerifyMode)
        {
            gameStarted = true; // Set the game started state
            PlayerPrefs.SetInt("isPaused", 0); // Save the paused state
            Time.timeScale = 1f; // Resume the game
        } else
        {
            PlayerPrefs.SetInt("isPaused", 1); // Save the paused state
            Time.timeScale = 0f; // Stop the game by default. 
        }
        Debug.Log("Init Time scale: " + Time.timeScale);
    }

    public void ResetTimer()
    {
        elapsedTime = 0f; // Reset the timer
        UpdateTimerDisplay(); // Update the timer display
    }

    public void SetGameEnded(bool ended)
    {
        gameEnded = ended; // Set the game ended state
    }

    public void SetGameStarted(bool started)
    {
        gameStarted = started; // Set the game started state
        if (started)
        {
            PlayerPrefs.SetInt("isPaused", 0); // Save the paused state
            Time.timeScale = 1f; // Resume the game
            Manager_Camera managerCamera = FindObjectOfType<Manager_Camera>();
            managerCamera.SetPositionToTarget(); // Set the camera to the target position
        }
        else
        {
            PlayerPrefs.SetInt("isPaused", 1); // Save the paused state
            Time.timeScale = 0f; // Pause the game
        }

        isPaused = !started; // Set the paused state
        //Debug.Log("Game started: " + started + " Time scale: " + Time.timeScale);
    }

    void Update()
    {
        if (gameEnded) return; // Do not update the timer if the game has ended
        // Update the timer if the game has started and is not paused
        if (gameStarted && !isPaused)
        {
            elapsedTime += Time.deltaTime; // Increment elapsed time
            UpdateTimerDisplay(); // Update the timer display
        }
    }

    public float GetElapsedTime()
    {
        return elapsedTime;
    }

    // Function to stop the game
    public void Stop()
    {
        if (!isPaused)
        {
            Time.timeScale = 0f; // Pause the game
            isPaused = true; // Update the paused state
            PlayerPrefs.SetInt("isPaused", 1); // Save the paused state
        }
    }

    // Function to resume the game
    public void Resume()
    {
        if (isPaused)
        {
            Time.timeScale = 1f; // Resume the game
            isPaused = false; // Update the paused state
            PlayerPrefs.SetInt("isPaused", 0); // Save the paused state
        }
    }

    // Optional: A toggle function to switch between stop and resume
    public void TogglePause()
    {
        if (isPaused)
        {
            Resume();
        }
        else
        {
            Stop();
        }
    }

    // Function to start the game and begin the timer
    public void StartGame()
    {
        if (!gameStarted)
        {
            gameStarted = true;
            elapsedTime = 0f; // Reset the timer
            Time.timeScale = 1f; // Ensure the game runs at normal speed

            Manager_Camera managerCamera = FindObjectOfType<Manager_Camera>();
            managerCamera.SetPositionToTarget(); // Set the camera to the target position
        }
    }

    // Function to update the timer display
    private void UpdateTimerDisplay()
    {
        int minutes = Mathf.FloorToInt(elapsedTime / 60); // Calculate minutes
        int seconds = Mathf.FloorToInt(elapsedTime % 60); // Calculate seconds
        timerText.text = string.Format("{0:00}:{1:00}", minutes, seconds); // Format and display the timer
    }

    public string FloatToMinutes(float time)
    {
        int minutes = Mathf.FloorToInt(time / 60); // Calculate minutes
        int seconds = Mathf.FloorToInt(time % 60); // Calculate seconds
        return string.Format("{0:00}:{1:00}", minutes, seconds); // Format and display the timer
    }
}
