using UnityEngine;
using UnityEngine.SceneManagement;
using System.Collections.Generic;
using System.Collections;

public class Manager_SceneTransitor : MonoBehaviour
{
    // The different scenes in the game
    public enum Scene
    {
        MainMenu,
        LevelEditor,
        Level,
        Tutorial
    }

    // The names of the scenes
    private Dictionary<Scene, string> sceneNames = new Dictionary<Scene, string>
    {
        { Scene.MainMenu, "1_Menu_Main" },
        { Scene.LevelEditor, "Level_Editor"},
        { Scene.Level, "Island_1_1"},
        { Scene.Tutorial, "Tutorial"}
    };

    public Manager_SceneFade sceneFade;
    private float defaultFadeTime = 0.5f;

    private void Awake()
    {
        sceneFade = FindObjectOfType<Manager_SceneFade>();
    }

    // Get the current scene
    public Scene GetCurrentSceneEnum()
    {
        string activeSceneName = SceneManager.GetActiveScene().name;

        foreach (var pair in sceneNames)
        {
            if (pair.Value == activeSceneName)
            {
                return pair.Key; // Return the matching Scene enum
            }
        }

        Debug.LogWarning("Scene not found in dictionary: " + activeSceneName);
        return Scene.MainMenu; // Return a default value if no match is found
    }

    // Loads a scene
    public void LoadScene(Scene scene)
    {
        if (sceneNames.TryGetValue(scene, out string sceneName))
        {
            UnityEngine.SceneManagement.SceneManager.LoadScene(sceneName);
        }
        else
        {
            Debug.LogError("Scene not found");
        }
    }

    // Reload the current scene
    public void ReloadScene()
    {
        StartCoroutine(LoadLevelCoroutine(GetCurrentSceneEnum(), true));
    }
    
    // Load the main menu
    public void LoadMainMenu()
    {
        StartCoroutine(LoadLevelCoroutine(Scene.MainMenu, false));
    }
    
    // Load the level editor
    public void LoadLevelEditor()
    {
        PlayerPrefs.SetInt("isVerifyMode", 0); // Load the map as editor
        PlayerPrefs.SetInt("isDownloaded", 0);
        StartCoroutine(LoadLevelCoroutine(Scene.LevelEditor, true));
    }

    // Load the Level in play mode
    public void LoadLevelEditorPlay()
    {
        PlayerPrefs.SetInt("isVerifyMode", 1); // Load the map as real test
        PlayerPrefs.SetInt("isDownloaded", 0);
        StartCoroutine(LoadLevelCoroutine(Scene.LevelEditor, true));
        
    }

    // Load the level
    public void LoadDownloadedLevel()
    {
        PlayerPrefs.SetInt("isVerifyMode", 1); // Load the map as real test
        PlayerPrefs.SetInt("isDownloaded", 1);
        StartCoroutine(LoadLevelCoroutine(Scene.LevelEditor, true));
    }

    // Load level
    public void LoadLevel()
    {
        StartCoroutine(LoadLevelCoroutine(Scene.Level, true));
    }

    // Load the tutorial
    public void LoadTutorial()
    {
        StartCoroutine(LoadLevelCoroutine(Scene.Tutorial, true));
    }

    IEnumerator LoadLevelCoroutine(Scene scene, bool forwardFade)
    {
        Time.timeScale = 1f; // Resume the game
        sceneFade.SceneFade(forwardFade);
        yield return new WaitForSeconds(0.5f);
        sceneFade.OnFadeComplete(forwardFade);
        LoadScene(scene);
    }

    // Quit the game
    public void QuitGame()
    {
        Application.Quit();
    }
}
