using UnityEngine;
using TMPro;
using UnityEngine.UI;

// Handles the tutorial flow
public class TutorialHandler : MonoBehaviour
{
    [SerializeField] private TextMeshProUGUI tutorialText;
    [SerializeField] private TextMeshProUGUI nextButtonText;    
    [SerializeField] private PositionIndicator positionIndicator;
    [SerializeField] private PositionIndicator moveIndicator;
    [SerializeField] private GameObject cameraResetButton;
    [SerializeField] private GameObject goalIndicator;
    [SerializeField] private GameObject TutorialTextIndicator;
    [SerializeField] private GameObject GameTimeIndicator;
    [SerializeField] private Button nextButton;
    [SerializeField] private Button previousButton;
    [SerializeField] private Manager_Score mng_level;
    [SerializeField] private PanZoom panZoom;
    private string[] tutorialTexts = new string[]
    {
        "Drag INSIDE the circle to aim and hit the ball. To cancel the shot, drag the ball back to the center.",
        "Drag OUTSIDE the circle to move the camera or pinch to zoom in and out.",
        "You can always reset the camera touching this button.",
        "Your objective is to reach the hole in the fewest shots possible! Good luck!",
        ""
    };

    [SerializeField] private int currentStep = 0;

    private void Awake()
    {
        positionIndicator.gameObject.SetActive(false);
        moveIndicator.gameObject.SetActive(false);
        cameraResetButton.SetActive(false);
        goalIndicator.SetActive(false);
        ShowTutorialStep(currentStep);
        positionIndicator.gameObject.SetActive(true);
        moveIndicator.gameObject.SetActive(true);

        // Go to the next step when the player strokes the ball
        mng_level.OnStrokeChanged += (isStroke) =>
        {   
            if (currentStep == 0)
            {
                currentStep = 1;
                ShowTutorialStep(currentStep);
            }            
        };

        // Go to the next step when the player moves the camera
        panZoom.OnCameraMoved += (isCameraMoved) =>
        {
            if (isCameraMoved)
            {
                if (currentStep == 1)
                {
                    currentStep = 2;
                    ShowTutorialStep(currentStep);
                }
            }
        };
        // Go to the next step when the player presses the reset button
        panZoom.OnResetCalled += (isCameraMoved) =>
        {
            if (currentStep == 2)
            {
                currentStep = 3;
                ShowTutorialStep(currentStep);
            }
        };
    }

    private void ShowTutorialStep(int step)
    {
        tutorialText.text = tutorialTexts[step];

        // Activate the helper based on the step
        positionIndicator.SetDisabled(step != 0);
        moveIndicator.SetDisabled(step != 1);
        cameraResetButton.SetActive(step == 2);
        goalIndicator.SetActive(step >= 3);
        nextButtonText.text = step == 3 ? "Hide Tutorial" : "Next";
        if (step > 3)
        {
            nextButtonText.text = "Finish Tutorial";
        }
        // Don't allow the go under the index
            previousButton.interactable = step != 0;

        // Hide the tutorial indicator text and display the time indicator of a normal level
        bool isTutorialEnd = (step == tutorialTexts.Length - 1);
        GameTimeIndicator.SetActive(isTutorialEnd);
        TutorialTextIndicator.SetActive(!isTutorialEnd);
        nextButton.interactable = !isTutorialEnd;
    }

    public void PreviousStep()
    {
        currentStep--;
        if (currentStep < 0)
        {
            currentStep = 0;
            return;
        }
        ShowTutorialStep(currentStep);
    }

    public void NextStep()
    {
        currentStep++;
        if (currentStep >= tutorialTexts.Length)
        {
            currentStep = tutorialTexts.Length - 1;
            return;
        }
        ShowTutorialStep(currentStep);
    }
}