using UnityEngine;
using UnityEngine.EventSystems;
using TMPro;
using RDG;

public class AimingSystem : MonoBehaviour, IPointerDownHandler, IDragHandler, IPointerUpHandler
{
    [SerializeField] private PlayerController playerController;
    [SerializeField] private Manager_Camera cameraManager; // Reference to the camera manager
    public RectTransform aimingPickUpArea; // The circular area where aiming is activated
    public RectTransform aimingCircle; // The circle that follows the pointer inside the aiming area
    public GameObject cancelZone; // Prefab for the aiming circle
    [SerializeField] private float maxRadius = 300f; // Max radius of the aiming circle
    [SerializeField] private float cancelThreshold = 50f; // Threshold to cancel the aiming
    private Vector2 shootDirection; // Direction of the aim
    [SerializeField] private TextMeshProUGUI shootPowerIndicator; // The power the ball will be shooted
    [SerializeField] private float distanceFromCenter;
    [SerializeField] private bool isAiming = false; // Whether the aiming mode is active
    private bool cancelShoot = false; // Whether the aiming is canceled
    public GameObject tempChampion; // TODO, remove
    public GameObject powerIndicator;
    private void Awake()
    {
        cameraManager = FindObjectOfType<Manager_Camera>();
        if (cameraManager == null)
        {
            Debug.LogError("Manager_Camera not found in the scene.");
        }
    }

    void Start()
    {
        cancelShoot = false;
        // Don't show cancel zone at the start
        cancelZone.SetActive(false);
        // Hide the aiming circle at the start
        aimingCircle.gameObject.SetActive(false);
        // Open the pickup area
        aimingPickUpArea.gameObject.SetActive(true);

        // Set the aiming area size based on the max radius
        maxRadius = GetPixelRadiusOfAimingArea(aimingCircle);
        cancelThreshold = GetPixelRadiusOfAimingArea(cancelZone.GetComponent<RectTransform>());
        // Find the player controller so we can shoot the balls
        playerController = GameObject.Find("Player").GetComponent<PlayerController>();
    }

    // This function will calculate the actual screen space size of the aiming area
    private float GetPixelRadiusOfAimingArea(RectTransform aimingPickUpArea)
    {
        // Get the width of the RectTransform (in local space, relative to its parent)
        float width = aimingPickUpArea.rect.width;

        // Get the canvas scale factor (to account for different screen sizes and resolutions)
        Canvas canvas = aimingPickUpArea.GetComponentInParent<Canvas>();
        float scaleFactor = canvas.scaleFactor;

        // Adjust the width by the scale factor to get the true screen-space size
        float pixelWidth = width * scaleFactor;

        // The radius is half the width (since it's a circle)
        float radiusInPixels = pixelWidth / 2;
        return radiusInPixels;
    }

    private void UpdateDistanceFromCenter(PointerEventData eventData)
    {
        maxRadius = GetPixelRadiusOfAimingArea(aimingCircle);
        cancelThreshold = GetPixelRadiusOfAimingArea(cancelZone.GetComponent<RectTransform>());
        // Get the center of the aimingPickUpArea (in screen space)
        Vector2 aimingAreaCenter = aimingPickUpArea.position;

        // Convert the center of the aiming area to screen space
        Vector2 screenAimingAreaCenter = RectTransformUtility.WorldToScreenPoint(Camera.main, aimingAreaCenter);

        // Now check the position relative to the center of the aiming area (circle)
        Vector2 relativePosition = eventData.position - screenAimingAreaCenter;
        distanceFromCenter = relativePosition.magnitude;

        /* Debug.Log("Update distance to center");
        Debug.Log(distanceFromCenter + " " + maxRadius); */

        // Calculate the direction of the aim
        shootDirection = relativePosition.normalized;
        bool isCanceling = distanceFromCenter <= cancelThreshold;

        // Update the power indicator based on the distance from the center
        float normalizedShootPower = NormalizeDistanceFromCenter();
        shootPowerIndicator.color = isCanceling ? Color.red : Color.white; // Change color to red if canceling
        shootPowerIndicator.text = $"{Mathf.RoundToInt(normalizedShootPower * 100)}%";

        if (isCanceling)
        {
            if (!cancelShoot)
            {
                // Trigger vibration if canceling
                TriggerVibration();
            }           
        }
        else
        {
            if (cancelShoot)
            {
               TriggerVibration(); // Trigger vibration if canceling
            }
            // Reset the vibration state
            cancelShoot = false;
        }

        // Update the player controller with the aiming state
        playerController.SetAimerState(!isCanceling);
        playerController.SetAimerVisible(!isCanceling);
        playerController.SetAimerDirection(shootDirection);
        playerController.SetAimerSize(normalizedShootPower);

        // This part may be removed? (the text, okay, but the temp champion?)
        // Set the champion to the max touch point position
        // Clamp distance between cancelThreshold and maxRadius
        float clampedDistance = Mathf.Clamp(distanceFromCenter, cancelThreshold, maxRadius);

        // Calculate the final screen position for the champion
        Vector2 finalScreenPosition = screenAimingAreaCenter + shootDirection * clampedDistance;

        // Convert the screen position to world position
        Vector3 worldPosition;
        RectTransformUtility.ScreenPointToWorldPointInRectangle(aimingPickUpArea, finalScreenPosition, Camera.main, out worldPosition);

        // Set the position of the temp champion
        tempChampion.transform.position = worldPosition;

        Vector2 perp = new Vector2(shootDirection.y, -shootDirection.x);
        finalScreenPosition = screenAimingAreaCenter + perp * (cancelThreshold + 50);
        RectTransformUtility.ScreenPointToWorldPointInRectangle(aimingPickUpArea, finalScreenPosition, Camera.main, out worldPosition);

        powerIndicator.transform.position = worldPosition;
        if (!isAiming)
        {
            powerIndicator.SetActive(false);
        }
        else
        {
            powerIndicator.SetActive(true);         
        }
    }

    // Call this method when you want to trigger vibration
    public void TriggerVibration()
    {
        cancelShoot = true; // Set cancel shoot to true
        // Check if the device supports vibration
        if (SystemInfo.supportsVibration)
        {
            Vibration.Vibrate(15); // Vibrate for 15 milliseconds
        }
    }

    public void OnPointerDown(PointerEventData eventData)
    {
        /* Debug.Log("Aiming started"); */
        cancelShoot = false; // Reset cancel shoot state
        isAiming = true;
        aimingPickUpArea.gameObject.SetActive(false); // We have picked up the aiming area
        cancelZone.SetActive(true); // Show the cancel zone
        aimingCircle.gameObject.SetActive(true); // Show the aiming circle
        cameraManager.OnStartAiming(); // Prevent the player from moving while aiming
        // Update the distance from the center
        UpdateDistanceFromCenter(eventData);
    }

    public void OnDrag(PointerEventData eventData)
    {
        // Update the distance from the center
        UpdateDistanceFromCenter(eventData);
        cameraManager.OnStartAiming(); // Prevent the player from moving while aiming
    }

    public void OnPointerUp(PointerEventData eventData)
    {
        /* Debug.Log("Aiming released"); */
        UpdateDistanceFromCenter(eventData);
        playerController.SetAimerState(false); // Allow the player to move again
        // When the user releases, check if the aiming should be reset or finalized
        if (isAiming)
        {
            // Clamp the distance to the max radius
            distanceFromCenter = Mathf.Clamp(distanceFromCenter, 0, maxRadius);
            if (distanceFromCenter <= cancelThreshold)
            {
                // Cancel the aiming
                /* Debug.Log("Aiming canceled"); */
            }
            else
            {
                float normalizedShootPower = NormalizeDistanceFromCenter();
                // Finalize the aiming by shooting the ball
                playerController.ShootBall(shootDirection, normalizedShootPower);
                /* Debug.Log("Shooting with power: " + normalizedShootPower); */
            }
        }

        /* Debug.Log("Aiming ended"); */
        aimingCircle.gameObject.SetActive(false); // Hide the aiming circle
        cancelZone.SetActive(false); // Hide the cancel zone
        aimingPickUpArea.gameObject.SetActive(true); // Show the aiming area again
        powerIndicator.SetActive(false); //
        isAiming = false; // Reset aiming state
    }

    private float NormalizeDistanceFromCenter()
    {
        // Ensure the distance is within the range between cancelThreshold and maxRadius
        // Clamp the distance to make sure it's between cancelThreshold and maxRadius
        float clampedDistance = Mathf.Clamp(distanceFromCenter, cancelThreshold, maxRadius);

        // Normalize the clamped distance to a range between 0 and 1
        float normalizedValue = Mathf.InverseLerp(cancelThreshold, maxRadius, clampedDistance);

        return normalizedValue;
    }
}
