using UnityEngine;
using UnityEngine.Audio;
using System.Collections;
using UnityEngine.UI;
using System;

// -> The input is obtained from Assets/Plugin_Joystick_Pack/Scripts/Joysticks/FixedJoystick.cs
// This scripts is responsible to control the ball movement.
public class PlayerController : MonoBehaviour
{
    [SerializeField] private Rigidbody2D mybody;
    [SerializeField] private Collider2D myCollider;

    [SerializeField] private PlayerSkin playerSkin;

    [SerializeField] private float velocityThreshold;

    [SerializeField] private Vector2 velocity = new Vector2(0f, 0f);
    private Vector3 direction = Vector3.zero;//ref to direction in which the ball will move
    // Actions
    [SerializeField] private bool throwable = true;
    [SerializeField] private bool dying = false;
    [SerializeField] private bool isShooting = false; // If the player is shooting the ball, it cannot be thrown again.

    [SerializeField] private PhysicsMaterial2D currentMaterial; // Each other ball, may have different values.
    
    [SerializeField] private Image shootAreaInteractor;

    [SerializeField] private GameObject aimer;
    [SerializeField] private GameObject aimerHighlight;
    [SerializeField] private Image petImage;
    [SerializeField] private Animator playerAnimator;
    [SerializeField] private GameObject parChampion;
    
    private bool isAiming = false;

    [SerializeField] private Manager_Camera managerCamera;

    [SerializeField] private Manager_Score scoreManager;
    [SerializeField] private Manager_Champion championManager;
    [SerializeField] private Manager_Player playerManager;
    public float MaxShootPower = 65f; // Maximum power for the ball
    public float MinShootPower = 5f; // Minimum power for the ball
    public event Action<Collision2D> OnPlayerCollision;

    public enum BallTrigger
    {
        Appear,
        Die
    }

    private void Awake()
    {
        mybody = GetComponent<Rigidbody2D>();
        scoreManager = FindObjectOfType<Manager_Score>();
        playerManager = FindObjectOfType<Manager_Player>();
        managerCamera = FindObjectOfType<Manager_Camera>();
    }

    public bool IsAiming()
    {
        return isAiming;
    }

    public void SetTrigger(BallTrigger trigger)
    {
        playerAnimator.SetTrigger(trigger.ToString());
    }

    private void FixedUpdate()
    {
        if (dying) return;
        velocity = mybody.linearVelocity;

        // If the ball is still, it is throwable. If it is moving, it is not throwable
        if (velocity.sqrMagnitude <= velocityThreshold && !dying)
        {
            if (!throwable)
            {
                throwable = true;
                mybody.linearVelocity = Vector2.zero;
                mybody.constraints = RigidbodyConstraints2D.FreezeRotation;
            }
        }
        else
        {
            // Slow down ball
            SlowDownBall();
            throwable = false;
            mybody.constraints = RigidbodyConstraints2D.None;
        }
        shootAreaInteractor.enabled = aimer.activeSelf? false : throwable;
    }

    public bool IsThrowable()
    {
        return throwable;
    }

    public void SetDying(bool value)
    {
        SetDying(value, false);
    }

    public void SetDying(bool value, bool isNewLoad)
    {
        dying = value;
        if (isNewLoad) return; // Don't touch the camera if it is a new load
        managerCamera.SetExploring(!value);
    }

    public void SetAimerState(bool state)
    {
        isAiming = state;
        throwable = !state;
        //managerCamera.SetExploring(!state);
    }

    public void SetExploring(bool state)
    {
        managerCamera.SetExploring(state);
    }

    public void SetAimerVisible(bool state)
    {
        aimer.SetActive(state);
        petImage.enabled = state;
        parChampion.SetActive(!state);
    }

    public void SetAimerDirection(Vector2 direction)
    {
        // Calculate the opposite direction by inverting the vector
        Vector2 oppositeDirection = -direction;
        
        // Convert the opposite direction from Vector2 to a 2D angle (for rotation)
        float angle = Mathf.Atan2(oppositeDirection.y, oppositeDirection.x) * Mathf.Rad2Deg;

        // Set the aimer's rotation
        aimer.transform.rotation = Quaternion.Euler(0, 0, angle);
        gameObject.transform.rotation = Quaternion.Euler(0, 0, angle);
    }

    public void SetAimerSize(float magnitude)
    {
        // Set the aimer's scale based on the magnitude of the joystick input
        aimerHighlight.transform.localScale = new Vector3(magnitude * 6, 6, 1);
    }

    // Given a normalized direction, shoot the ball in that direction with the given power
    public void ShootBall(Vector2 direction, float magnitude)
    {
        if (!throwable) return;

        // If it was able to shoot, then position is valid, so add it to the list
        playerManager.AddLastValidPosition(mybody.position);

        // Calculate the power based on the normalized value
        float power = magnitude * (MaxShootPower - MinShootPower) + MinShootPower;

        bool hitStop = magnitude >= 0.99f; // Check if the power is at maximum
        //Debug.Log("Hit stop: " + hitStop + " magnitude: " + magnitude);
        //Debug.Log("Shooting ball with power: " + power);

        StartCoroutine(ExecuteHitSequence(direction, power, hitStop));
    }

    public IEnumerator ExecuteHitSequence(Vector2 direction, float power, bool hitStop)
    {
        throwable = false;
        // Move the par champion towards the ball and wait for completion
        yield return StartCoroutine(championManager.MoveToBall(hitStop));

        // Now continue with the next actions **after** the champion reaches the ball
        scoreManager.AddStrokes(1);
        SetAimerVisible(false); // Hide the aimer
        // Move the ball in the required direction
        mybody.linearVelocity = (-direction.normalized) * power;
        managerCamera.OnShotFired(); // Follow the ball
    }

    // Get the ball's rigidbody
    public Rigidbody2D GetRigidbody()
    {
        return mybody;
    }

    // Get the ball's collider
    public Collider2D GetCollider()
    {
        return myCollider;
    }

    // Allow other scripts to subscribe to the collision event
    private void OnCollisionEnter2D(Collision2D collision)
    {
        // Notify listeners (other scripts) about the collision
        OnPlayerCollision?.Invoke(collision);
    }


    // Slow downs the ball
    void SlowDownBall()
{
    // Define base deceleration factors
    float baseDeceleration = 0.15f;       // Base factor for linear motion
    float maxLinearMultiplier = 0.15f;       // Increase deceleration when slow
    float lowSpeedThreshold = 6f;       // Threshold for when to apply extra deceleration
    
    float angularDeceleration = 3f;       // Base factor for angular motion
    float maxAngularMultiplier = 2f;      // Increase deceleration for spin when low
    float lowAngularThreshold = 0.5f;     // Threshold for angular velocity
    
    // Handle linear deceleration
    Vector2 velocity = mybody.linearVelocity;
    float speed = velocity.magnitude;
    if (speed < 0.1f)
    {
        mybody.linearVelocity = Vector2.zero;
    }
    else
    {
        // Apply extra deceleration if speed is low
        if (speed < lowSpeedThreshold)
        {
            Vector2 deceleration = velocity.normalized * maxLinearMultiplier;
            velocity -= deceleration;
        }
        else
        {
            Vector2 deceleration = velocity.normalized * baseDeceleration * speed;
            velocity -= deceleration * Time.fixedDeltaTime;
        }

        mybody.linearVelocity = velocity;
        
    }

    // Handle angular deceleration similarly
    float angVel = mybody.angularVelocity;
    if (Mathf.Abs(angVel) < 0.1f)
    {
        mybody.angularVelocity = 0f;
    }
    else
    {
        float angularMultiplier = (Mathf.Abs(angVel) < lowAngularThreshold) ? maxAngularMultiplier : 1f;
        angVel -= Mathf.Sign(angVel) * angularDeceleration * angularMultiplier * Time.fixedDeltaTime;
        mybody.angularVelocity = angVel;
    }
}


}