using UnityEngine;
using UnityEngine.Audio;
using System.Collections;
using UnityEngine.UI;

// -> The input is obtained from Assets/Plugin_Joystick_Pack/Scripts/Joysticks/FixedJoystick.cs
// This scripts is responsible to control the ball movement.
public class PlayerControllerLite : MonoBehaviour
{
    [SerializeField] private Rigidbody2D mybody;

    [SerializeField] private PlayerSkin playerSkin;

    [SerializeField] private float velocityThreshold;
    [SerializeField] private float maxVelocity = 10f;

    [SerializeField] private Vector2 velocity = new Vector2(0f, 0f);
    private Vector3 direction = Vector3.zero;//ref to direction in which the ball will move
    // Actions
    [SerializeField] private bool throwable = true;
    [SerializeField] private bool dying = false;

    [SerializeField] private PhysicsMaterial2D currentMaterial; // Each other ball, may have different values.
    
    [SerializeField] private Image shootAreaInteractor;

    [SerializeField] private GameObject aimer;
    [SerializeField] private GameObject aimerHighlight;
    [SerializeField] private Image petImage;
    [SerializeField] private Animator playerAnimator;
    
    private bool isAiming = false;


    public enum BallTrigger
    {
        Appear,
        Die
    }

    private void Awake()
    {
        mybody = GetComponent<Rigidbody2D>();
    }

    public void SetTrigger(BallTrigger trigger)
    {
        playerAnimator.SetTrigger(trigger.ToString());
    }

    private void FixedUpdate()
    {
        if (dying) return;
        velocity = mybody.linearVelocity;

        if (velocity.sqrMagnitude > maxVelocity)
        {
            mybody.linearVelocity = Vector2.zero;
            mybody.constraints = RigidbodyConstraints2D.FreezeRotation;
        }

        // If the ball is still, it is throwable. If it is moving, it is not throwable
        if (velocity.sqrMagnitude <= velocityThreshold && !dying)
        {
            if (!throwable)
            {
                throwable = true;
                mybody.linearVelocity = Vector2.zero;
                mybody.constraints = RigidbodyConstraints2D.FreezeRotation;
            }
        }
        else 
        {
            // Slow down ball
            SlowDownBall();
            throwable = false;
            mybody.constraints = RigidbodyConstraints2D.None;
        }
        shootAreaInteractor.enabled = aimer.activeSelf? false : throwable;
    }

    public bool IsThrowable()
    {
        return throwable;
    }

    public void SetDying(bool value)
    {
        dying = value;
    }

    public void SetAimerState(bool state)
    {
        isAiming = state;
        throwable = !state;
    }

    public void SetAimerVisible(bool state)
    {
        aimer.SetActive(state);
        petImage.enabled = state;
    }

    public void SetAimerDirection(Vector2 direction)
    {
        // Calculate the opposite direction by inverting the vector
        Vector2 oppositeDirection = -direction;
        
        // Convert the opposite direction from Vector2 to a 2D angle (for rotation)
        float angle = Mathf.Atan2(oppositeDirection.y, oppositeDirection.x) * Mathf.Rad2Deg;

        // Set the aimer's rotation
        aimer.transform.rotation = Quaternion.Euler(0, 0, angle);
        gameObject.transform.rotation = Quaternion.Euler(0, 0, angle);
    }

    public void SetAimerSize(float magnitude)
    {
        // Set the aimer's scale based on the magnitude of the joystick input
        aimerHighlight.transform.localScale = new Vector3(magnitude * 6, 6, 1);
    }

    // Given a normalized direction, shoot the ball in that direction with the given power
    public void ShootBall(Vector2 direction, float power)
    {
        if (!throwable) return;

        //moves the ball in the required direction
        mybody.linearVelocity = (-direction.normalized) * power;
        throwable = false;
    }

    // Slow downs the ball
    void SlowDownBall()
    {
        //we stop smoothly Player
        if (mybody.linearVelocity.x > 0)
        {
            mybody.linearVelocity -= new Vector2(0.04f, 0);
        }
        if (mybody.linearVelocity.y > 0)
        {
            mybody.linearVelocity -= new Vector2(0, 0.04f);
        }
        if (mybody.linearVelocity.x < 0)
        {
            mybody.linearVelocity += new Vector2(0.015f, 0);
        }
        if (mybody.linearVelocity.y < 0)
        {
            mybody.linearVelocity += new Vector2(0, 0.015f);
        }
        if (mybody.angularVelocity > 0)
        {
            mybody.angularVelocity -= 3f;
        }
        if (mybody.angularVelocity < 0)
        {
            mybody.angularVelocity += 0.8f;
        }
    }
}