using UnityEngine;
using System.Collections.Generic;
using System;
using System.Linq;

public class PlayerEquipment : MonoBehaviour
{
  [SerializeField] Scriptable_ShopItemData shopItemData = null;
  [SerializeField] PlayerSkin playerSkin = null;

  Dictionary<int, bool> skins = new Dictionary<int, bool>() { { 0, true } };
  Dictionary<int, bool> trailEffects = new Dictionary<int, bool>() { { 1000, true } };

  [SerializeField] Manager_Auth authManager = null;
  [SerializeField] Manager_UserData userDataManager = null;
  private string key = "skins";

  [SerializeField] private Warning warning = null;
  [SerializeField] private bool isPreparationMenu = false;

  public void Awake()
  {
    // Load the purchased items if the player is logged in
    if (isPreparationMenu == false)
    {
      authManager.OnLoggedIn += LoadPurchasedItems;
      authManager.OnLoggedOut += ResetPurchasedItems;
    }
    {
      LoadPurchasedItems();
      LoadEquippedItems();
    }
  }

  private void ResetPurchasedItems()
  {
    Debug.Log("Resetting purchased items to default");
    skins.Clear();
    trailEffects.Clear();
    PlayerPrefs.SetInt("BallSkin", 0);
    PlayerPrefs.SetInt("TrailEffect", 1000);
    skins.Add(0, true);
    trailEffects.Add(1000, true);
    LocalSave();
    ReloadPurchasedItems();
    LoadEquippedItems();
  }

  public void ReloadPurchasedItems()
  {
    Object_ItemInteractor[] allItems = FindObjectsOfType<Object_ItemInteractor>(false);
    if (allItems.Length == 0)
    {
      return;
    }
    allItems[0].SetEquippedItem(PlayerPrefs.GetInt("BallSkin"));
    allItems[0].SetEquippedItem(PlayerPrefs.GetInt("TrailEffect"));
    foreach (Object_ItemInteractor item in allItems)
    {
      Debug.Log("Reloading item: " + item.itemName);
      item.UpdateItemState();
    }
  }

  // Load the equipped items
  public void LoadEquippedItems()
  {
    int ballSkin = PlayerPrefs.GetInt("BallSkin");
    int trailEffect = PlayerPrefs.GetInt("TrailEffect");

    EquipItem(Manager_Equip.EquipType.BallSkin, ballSkin);
    EquipItem(Manager_Equip.EquipType.TrailEffect, trailEffect);
  }

  // Save the item purchase
  public void SaveItemPurchase(Manager_Equip.EquipType equipType, int itemID)
  {
    switch (equipType)
    {
      case Manager_Equip.EquipType.BallSkin:
        if (skins.ContainsKey(itemID))
        {
          Debug.Log("Item already purchased");
          return;
        }
        skins.Add(itemID, true);
        break;
      case Manager_Equip.EquipType.TrailEffect:
        if (trailEffects.ContainsKey(itemID))
        {
          Debug.Log("Item already purchased");
          return;
        }
        trailEffects.Add(itemID, true);
        break;
    }
    SavePurchasedItems();
  }

  public void LocalSave()
  {
     // Save the purchased items
    ES3.Save(key, skins, ShopKeys.GetEquipPath(ShopKeys.SKIN_FILE));
    ES3.Save(key, trailEffects, ShopKeys.GetEquipPath(ShopKeys.TRAIL_FILE));
  }

  // Save the purchased items
  public void SavePurchasedItems()
  {
    // Save the purchased items
    LocalSave();

    if (authManager.IsUserLoggedIn())
    {
      Debug.Log("Saving purchased items to Firestore");
      userDataManager.TryUploadPurchases(skins, trailEffects);
    }
  }


  // Check if the item is purchased
  public bool IsItemPurchased(Manager_Equip.EquipType equipType, int itemID)
  {
    bool isPurchased = false;

    switch (equipType)
    {
      case Manager_Equip.EquipType.BallSkin:
        isPurchased = skins.ContainsKey(itemID);
        break;
      case Manager_Equip.EquipType.TrailEffect:
        isPurchased = trailEffects.ContainsKey(itemID);
        break;
    }

    return isPurchased;
  }

  // Load the purchased items
  public void LoadPurchasedItems()
  {
    if (authManager != null && authManager.IsUserLoggedIn())
    {
      Debug.Log("Loading purchased items from Firestore");
      LoadPurchasedItemsFromFirestore();
    }
    else
    {
      LoadPurchasedItemsFromLocal();
    }
  }

  // Load the purchased items from local storage
  private void LoadPurchasedItemsFromLocal()
  {
    if (ES3.KeyExists(key, ShopKeys.GetEquipPath(ShopKeys.SKIN_FILE)))
    {
      Debug.Log("Loading purchased skin");
      skins = ES3.Load<Dictionary<int, bool>>(key, ShopKeys.GetEquipPath(ShopKeys.SKIN_FILE));
    }
    if (ES3.KeyExists(key, ShopKeys.GetEquipPath(ShopKeys.TRAIL_FILE)))
    {
      Debug.Log("Loading purchased trail");
      trailEffects = ES3.Load<Dictionary<int, bool>>(key, ShopKeys.GetEquipPath(ShopKeys.TRAIL_FILE));
    }
  }

  // Load the purchased items from Firestore
  public void LoadPurchasedItemsFromFirestore()
  {
    if (authManager.IsUserLoggedIn())
    {
      // TODO use await??
      userDataManager.LoadPurchases(purchaseData =>
      {
          if (purchaseData != null)
          {
              if (purchaseData.TryGetValue(ShopKeys.SKIN_FILE, out object skinObj) &&
                  purchaseData.TryGetValue(ShopKeys.TRAIL_FILE, out object trailObj))
              {
                  // Convert back from Firestore format (string keys) to Dictionary<int, bool>
                  Dictionary<string, object> skinDictRaw = skinObj as Dictionary<string, object>;
                  Dictionary<string, object> trailDictRaw = trailObj as Dictionary<string, object>;

                  skins = skinDictRaw?.ToDictionary(
                      kvp => int.Parse(kvp.Key),
                      kvp => Convert.ToBoolean(kvp.Value)
                  );

                  trailEffects = trailDictRaw?.ToDictionary(
                      kvp => int.Parse(kvp.Key),
                      kvp => Convert.ToBoolean(kvp.Value)
                  );

                  // Also save to local storage and load it
                  LocalSave();
                  // Set the default items as equipped
                  PlayerPrefs.SetInt("BallSkin", 0);
                  PlayerPrefs.SetInt("TrailEffect", 1000);
                  LoadEquippedItems();
                  ReloadPurchasedItems();

                  Debug.Log("Loaded purchased items from Firestore");
              }
              else
              {
                LoadPurchasedItemsFromLocal();
              }
          }
          else
          {
              Debug.LogError("No data found in Firestore");
              // If Firestore load failed or has no data
              LoadPurchasedItemsFromLocal();
              warning.ShowSuccess("Welcome! Your local purchases are now linked to your account.");
              SavePurchasedItems();
          }
      });

    }
  }

  // Equip the item
  public void EquipItem(Manager_Equip.EquipType equipType, int itemID)
  {
    switch (equipType)
    {
      case Manager_Equip.EquipType.BallSkin:
        EquipBallSkin(playerSkin, itemID);
        break;
      case Manager_Equip.EquipType.TrailEffect:
        EquipTrailEffect(playerSkin, itemID);
        break;
    }
  }

  public void EquipBallSkin(PlayerSkin playerSkin, int itemID)
  {
    // Equip the ball skin
    PlayerPrefs.SetInt("BallSkin", itemID);
    playerSkin.SetPlayerSkin(shopItemData.GetBallSkin(itemID));
  }

  public void EquipTrailEffect(PlayerSkin playerSkin, int itemID)
  {
    // Equip the trail effect
    PlayerPrefs.SetInt("TrailEffect", itemID);
    playerSkin.SetPlayerTrail(shopItemData.GetTrailEffect(itemID));
  }

}