using UnityEngine;
using UnityEngine.UI;
using UnityEngine.EventSystems;
using TMPro;

public class UI_PlayerInteraction : MonoBehaviour
{
    public Image powerBar;  // UI element for the power bar
    public TextMeshProUGUI shootPowerIndicator; // The power the ball will be shooted

    public void SetPowerValue(float value)
    {
        // Set the power bar fill amount and update the text indicator
        powerBar.fillAmount = Mathf.Clamp01(value);
        shootPowerIndicator.text = $"{Mathf.RoundToInt(value * 100)}%";
    }
    
    public void SetPowerBarFill(float fillAmount)
    {
        // Set power bar fill amount and clamp it between 0 and 1
        powerBar.fillAmount = Mathf.Clamp01(fillAmount);
    }

    public void AdjustPowerBar(float amount)
    {
        // Adjust the power bar fill amount by the specified amount
        SetPowerBarFill(powerBar.fillAmount + amount);
    }

    public float GetPowerBarFill()
    {
        // Get the current fill amount of the power bar
        return powerBar.fillAmount;
    }

    public float GetNormalizedTouchPosition(PointerEventData eventData, bool useXAxis)
    {
        Vector2 touchPosition = eventData.position;

        // Convert the touch position to local position within the Image
        Vector2 localPosition;
        RectTransformUtility.ScreenPointToLocalPointInRectangle(
            powerBar.rectTransform,
            touchPosition,
            eventData.pressEventCamera,
            out localPosition
        );

        // Normalize the local position within the image bounds
        Vector2 normalizedTouchPosition = NormalizeTouchPosition(localPosition, powerBar.rectTransform.rect.size);
        float normPos = useXAxis ? normalizedTouchPosition.x : normalizedTouchPosition.y;
        return Mathf.Floor(normPos * 100) / 100;  // Round to 2 decimal places
    }

    private Vector2 NormalizeTouchPosition(Vector2 touchPosition, Vector2 imageSize)
    {
        // Normalize the touch position to the range of 0 to 1 within the image bounds
        float normalizedX = Mathf.Clamp01(touchPosition.x / imageSize.x);
        float normalizedY = Mathf.Clamp01(touchPosition.y / imageSize.y);
        return new Vector2(normalizedX, normalizedY);
    }
}
