using TMPro;
using UnityEngine;
using UnityEngine.UI;
using System.Collections.Generic;

// Handles the Display of the maps
public class UI_MapDataDisplay : MonoBehaviour
{
    /***** UI ******/
    [SerializeField] private TMP_InputField initialName;
    [SerializeField] private TMP_InputField editedName;
    [SerializeField] private TMP_InputField editedDescription;

    /***** Menus ******/
    [SerializeField] private GameObject interactionMenu;
    [SerializeField] private GameObject statusText; // Disables when there are maps, enables when there are no maps which shows a message
    [SerializeField] private Transform mapsViewport; // Reference to the viewport where maps are displayed

    [SerializeField] private TextMeshProUGUI lastModDateLabel;
    [SerializeField] private TextMeshProUGUI avgPar;
    [SerializeField] private TextMeshProUGUI totalHoles;
    [SerializeField] private TextMeshProUGUI fileSize;

    [SerializeField] private GameObject verifiedIndicator;
    [SerializeField] private GameObject unverifiedIndicator;

    [SerializeField] private GameObject sharedIndicator;
    [SerializeField] private GameObject unsharedIndicator;

    public void GetInputFields(out TMP_InputField name, out TMP_InputField description)
    {
        name = editedName;
        description = editedDescription;
    }

    // Create a new map
    public string GetMapInitialName()
    {
        return initialName.text;
    }

    public void SetInteractionMenuStatus(bool open)
    {
        interactionMenu.SetActive(open);
    }

    // Show the map editing menu
    public void ShowMapEdit(MapData_ES3 mapData)
    {
        Debug.Log("Editing map: " + mapData.name);
        SetInteractionMenuStatus(true); // Open the interaction menu

        // Update the interaction menu with the map data
        editedName.text = mapData.name;
        editedDescription.text = mapData.description;
        UpdateTitleState(mapData.isUploaded);
        
        // Set the current map name for creating the new map correctly if needed
        PlayerPrefs.SetString(KeyManager.saveKeysDict[KeyManager.SaveKey.EditMapKey], mapData.key);

        // Set the verified state and thus, if we can share the map
        verifiedIndicator.SetActive(mapData.isValidated);
        unverifiedIndicator.SetActive(!mapData.isValidated);

        // Update other labels data
        sharedIndicator.SetActive(mapData.isUploaded);
        unsharedIndicator.SetActive(!mapData.isUploaded);
        lastModDateLabel.text = mapData.lastModifiedDate;

        avgPar.text = "Avg. Par: " + MapData_ES3.GetAveragePar(mapData.holesPar).ToString("F2");
        totalHoles.text = "Holes: " + mapData.holesPar.Count.ToString();
        fileSize.text = "Size:\n" + mapData.fileSize;
    }

    public void UpdateTitleState(bool isUploaded)
    {
        editedName.interactable = !isUploaded;
        // Change the color of the text to golden if the map is uploaded
        if (isUploaded)
        {
            editedName.textComponent.color = new Color(1, 1, 0, 1);
        }
        else
        {
            editedName.textComponent.color = new Color(1, 1, 1, 1);
        }
    }

    // Show a warning message
    private void Update()
    {
        statusText.SetActive(mapsViewport.childCount == 0);
    }

}


