using TMPro;
using UnityEngine;
using UnityEngine.UI;

// Handles the Display of the maps
public class UI_MapDataDisplayCloud : MonoBehaviour
{
    /***** UI ******/
    [SerializeField] private TMP_InputField mapNameText;
    [SerializeField] private TMP_InputField mapDescriptionText;
    [SerializeField] private TextMeshProUGUI authorNameText;
    [SerializeField] private TextMeshProUGUI downloadButtonText;
    [SerializeField] private Button downloadButton;
    [SerializeField] private Button removeButton;
    [SerializeField] private Button updateButton;
    [SerializeField] private Button rateButton;
    [SerializeField] private GameObject[] ratingStars;
    [SerializeField] private Button voteButton;
    [SerializeField] private GameObject interactionMenu;

    // Can be refactored into interface or abstract class
    [SerializeField] private Transform mapsViewport; // Reference to the viewport where maps are displayed

    [SerializeField] private TextMeshProUGUI lastModDateLabel;
    [SerializeField] private TextMeshProUGUI avgPar;
    [SerializeField] private TextMeshProUGUI totalHoles;
    [SerializeField] private TextMeshProUGUI fileSize;
    [SerializeField] private TextMeshProUGUI downloadText;
    [SerializeField] private TextMeshProUGUI likeText;
    [SerializeField] private TextMeshProUGUI avgRatingText;
    [SerializeField] private TextMeshProUGUI authorID;
    [SerializeField] private TextMeshProUGUI authorName;
    [SerializeField] private Warning warning;

    // Star rating UI
    [SerializeField] private Image[] starImages;
    [SerializeField] private Sprite starOn;
    [SerializeField] private Sprite starHalf;
    [SerializeField] private Sprite starOff;

    public enum VoteValue
    {
        Up = 1,
        Down = -1
    }

    public enum RateValue
    {
        One = 1,
        Two = 2,
        Three = 3,
        Four = 4,
        Five = 5
    }

    public VoteValue voteValue = VoteValue.Up;
    public RateValue rateValue = RateValue.Five;

    // Copy the user ID to the clipboard
    public void CopyUserID()
    {
        GUIUtility.systemCopyBuffer = authorID.text;
        warning.ShowWarning("User ID copied to clipboard!", Warning.WarningType.Positive);
        Debug.Log("User ID copied to clipboard: " + authorID.text);
    }

    // Show the map editing menu
    public void ShowMapInteractor(MapData_Cloud data)
    {
        SetInteractionMenuState(true);
        mapNameText.text = data.mapName;
        mapDescriptionText.text = data.mapDescription;
        authorNameText.text = "Author:" + data.mapAuthor;
        lastModDateLabel.text = data.lastModifiedDate;

        avgPar.text = "Avg. Par: " + MapData_ES3.GetAveragePar(data.holesPar).ToString("F2");
        totalHoles.text = "Holes: " + data.holesPar.Count.ToString();
        fileSize.text = "Size:\n" + data.fileSize;

        downloadText.text = $"Downloads:\n {data.downloads}";
        likeText.text = $"Likes:\n {data.likes}";
        float avgRating = data.rates > 0 ? (float)data.totalRating / data.rates : 0f;
        if (avgRating == 0f)
        {
            avgRatingText.text = "Be the first to rate!";
        }
        else
        {
            avgRatingText.text = $"Avg.: {avgRating:0.00}/5 from {data.rates} rating{(data.rates == 1 ? "" : "s")}";
        }

        UpdateStarRating(avgRating);

        PlayerPrefs.SetString(KeyManager.saveKeysDict[KeyManager.SaveKey.EditMapKey], data.key);

        authorID.text = data.mapAuthorID;
        authorName.text = data.mapAuthor;
    }

    private void UpdateStarRating(float rating)
    {
        // The ratings is between 1 to 5, and can take half values
        int fullStars = Mathf.FloorToInt(rating);
        float fraction = rating - fullStars;
        int halfStars = fraction >= 0.5f ? 1 : 0;
        int emptyStars = 5 - fullStars - halfStars;
        for (int i = 0; i < starImages.Length; i++)
        {
            if (i < fullStars)
            {
                starImages[i].sprite = starOn;
            }
            else if (i < fullStars + halfStars)
            {
                starImages[i].sprite = starHalf;
            }
            else
            {
                starImages[i].sprite = starOff;
            }
        }   
    }

    // Set the interaction button text and interactability
    public void SetInteractionButton(bool isDownloaded)
    {
        Debug.Log("Setting button text");
        downloadButtonText.text = isDownloaded ? "Play" : "Download"; //TODO -> constant key
        // Only allow to interact with the buttons if the map is downloaded
        removeButton.interactable = isDownloaded;
        updateButton.interactable = isDownloaded;
        rateButton.interactable = isDownloaded;
        voteButton.interactable = isDownloaded;    
    }

    public void SetInteractionMenuState(bool state)
    {
        interactionMenu.SetActive(state);
    }

    // Set the vote value
    public void SetVoteValue(int value)
    {
        
        voteValue = value switch
        {
            1 => VoteValue.Up,
            -1 => VoteValue.Down,
            _ => voteValue
        };
    }

    // Set the rate value
    public void SetRateValue(int value)
    {
        rateValue = value switch
        {
            1 => RateValue.One,
            2 => RateValue.Two,
            3 => RateValue.Three,
            4 => RateValue.Four,
            5 => RateValue.Five,
            _ => rateValue
        };
        for (int i = 0; i < ratingStars.Length; i++)
        {
            ratingStars[i].SetActive(i < (int)rateValue);
        }
    }

    public int GetVoteValue()
    {
        return (int)voteValue;
    }
    public int GetRateValue()
    {
        return (int)rateValue;
    }
}


