using UnityEngine;
using System.Collections.Generic;
using System.IO;

public class CRUD_EasySave3 : MonoBehaviour
{
    // TODO consider using bytes to save the data as it is more efficient
    // Save the map data by compressing it
    public long SaveMapData(List<TileMapData> tilesData)
    {
        // Get the map name, key, and paths
        LocalSaveKeys.GetLocalSaveInfo(out var mapName, out var mapKey, 
                                                out var mapPath, out var tempPath, LocalSaveKeys.SaveType.Local);
        
        // Save the raw tile map data using ES3 in JSON format
        ES3.Save(mapKey, tilesData, tempPath);

       // At this point, `tempPath` is the path where the data was saved
        Debug.Log("Data saved to: " + tempPath);
        
        string filePath = Path.Combine(Application.persistentDataPath, tempPath);

        // Check the file size using the `tempPath`
        FileInfo fileInfo = new FileInfo(filePath);
        long fileSizeBytes = fileInfo.Length;
        Debug.Log("File size: " + CompressionUtility.FormatFileSize(fileSizeBytes));
        // Loads the raw data and compresses it
        string mapData = ES3.LoadRawString(tempPath);
        string compressedMapData = CompressionUtility.CompressString(mapData);

        // Save the compressed map data and delete the decompressed file
        ES3.DeleteFile(tempPath);
        ES3.SaveRaw(compressedMapData, mapPath);

        // Return the (uncompressed) file size
        return fileSizeBytes;
    }

    // Check if the map exists
    public bool CheckMapExists(string mapPath)
    {
        return ES3.FileExists(mapPath);
    }

    // Load the map data
    public List<TileMapData> LoadMapData(LocalSaveKeys.SaveType saveType)
    {
        // Get the map name, key, and paths
        LocalSaveKeys.GetLocalSaveInfo(out var mapName, out var mapKey, 
                                                out var mapPath, out var tempPath, saveType);

        Debug.Log("Loading map data from " + mapPath);
        if(CheckMapExists(mapPath) == false)
        {
            Debug.Log("Map doesn't exist");
            return new List<TileMapData>();
        }
        
        // Load and uncompressed the saved tile map data
        string mapData = ES3.LoadRawString(mapPath);
        string decompressedData = CompressionUtility.DecompressString(mapData);

        // Use the temporary path to save the decompressed data
        ES3.SaveRaw(decompressedData, tempPath);

        // Load the data using ES3 and remove the temporary file as we don't need it anymore
        List<TileMapData> tileMapData = ES3.Load(mapKey, tempPath, new List<TileMapData>());
        ES3.DeleteFile(tempPath);

        return tileMapData;
    }

    // Loads the raw data of a map
    public string LoadRawMapData(string name, LocalSaveKeys.SaveType saveType)
    {
        return ES3.LoadRawString(LocalSaveKeys.GetMapPath(name, saveType));
    }

    // Saves the raw data of a map in a given path
    public void SaveRawMapData(string name, string data, LocalSaveKeys.SaveType saveType)
    {
        ES3.SaveRaw(data, LocalSaveKeys.GetMapPath(name, saveType));
    }

    // Delete the map data
    public void DeleteMapData(string mapName, LocalSaveKeys.SaveType saveType)
    {
        ES3.DeleteFile(LocalSaveKeys.GetMapPath(mapName, saveType));
    }

    // Rename the map data
    public void RenameMap(string originalName, string newName, LocalSaveKeys.SaveType saveType)
    {
        string originalMapPath = LocalSaveKeys.GetMapPath(originalName, saveType);
        string newMapPath = LocalSaveKeys.GetMapPath(newName, saveType);
        Debug.Log("Renaming map from " + originalMapPath + " to " + newMapPath);
        ES3.RenameFile(originalMapPath, newMapPath);
    }

    // Create a copy of the map
    public void CreateCopyOfMap(string originalName, string newName, LocalSaveKeys.SaveType saveType)
    {
        string originalMapPath = LocalSaveKeys.GetMapPath(originalName, saveType);
        string newMapPath = LocalSaveKeys.GetMapPath(newName, saveType);

        ES3.CopyFile(originalMapPath, newMapPath);
    }

    /*** Acces to the file which indexes all the maps ***/

    // Save the listing info of the maps
    public void SaveListingInfo(Dictionary<string, IMapData> maps, LocalSaveKeys.SaveType saveType)
    {
        string saveKey = LocalSaveKeys.MAIN_KEY;
        string mapPath = LocalSaveKeys.GetIndexPath(saveType);
        
        ES3.Save(saveKey, maps, mapPath);
    }

    // Load the listing info of the maps
    public Dictionary<string, IMapData> LoadMapListingInfo(LocalSaveKeys.SaveType saveType)
    {
        string saveKey = LocalSaveKeys.MAIN_KEY;
        string mapPath = LocalSaveKeys.GetIndexPath(saveType);

        Dictionary<string, IMapData> maps = ES3.Load(saveKey, mapPath, new Dictionary<string, IMapData>());

        return maps;
    }
}