using UnityEngine;
using System;

// Handles the interaction with the map data -> TODO error handling
public class Manager_MapDataCloud : MonoBehaviour
{
    [SerializeField] private Interact_MapDataDisplayCloud processMapDataDisplay; // Reference to the info / processing of the display of the map data
    [SerializeField] private UI_MapDataDisplayCloud uiMapDataDisplay; // Reference to the map menus manager script
    [SerializeField] private Manager_IndexData localIndexSave; // Reference to the local save script
    [SerializeField] private Manager_TilesData tilesDataSave; // Reference to the map tiles manager script
    [SerializeField] private Manager_SceneTransitor sceneTransitorManager; // Reference to the scene transitor manager script
    [SerializeField] private Manager_MapAccess cloudDownload; // Reference to the cloud download manager script
    [SerializeField] private Manager_UserData userDataManager; // Reference to the user data manager script
    [SerializeField] private CRUD_EasySave3_Cooldown cooldown; // Reference to the cooldown manager script
    [SerializeField] private Warning warning;
    // Data origin
    private LocalSaveKeys.SaveType saveType = LocalSaveKeys.SaveType.Download; // The save type of the map data

    private void Awake()
    {
        // Load the existing maps and display them
        processMapDataDisplay.SetMaps(localIndexSave.GetMapListingInfo(saveType));
        processMapDataDisplay.LoadMaps();
        Debug.Log("Loaded maps: " + processMapDataDisplay.GetMaps().Count);
    }

    /***** UI interaction ****/

    // Load the map for playing
    public void EnterGameMode()
    {
        UpdateMapData();
        Debug.Log("Loading map for play: " + processMapDataDisplay.GetCurrentMap().mapName);
        sceneTransitorManager.LoadDownloadedLevel();
    }

    // Edit the map maybe in the future after having a permission system
    /* public void EnterEditMode()
    {
        UpdateMapData();
        Debug.Log("Loading map editing scene: " + processMapDataDisplay.GetCurrentMap().name);
        sceneTransitorManager.LoadLevelEditor();
    } */

     // Add data to the interaction menu
    public void PopoulateInteractionMenu(MapData_Cloud mapData)
    {
        uiMapDataDisplay.ShowMapInteractor(mapData);
        processMapDataDisplay.SetCurrentMap(mapData);
        uiMapDataDisplay.SetInteractionButton(UpdateMapData());
    }

    /**** CRUD operations ****/
    public void DownloadMap()
    {
        MapData_Cloud mapDataCloud = processMapDataDisplay.GetCurrentMap();

        string mapPath = mapDataCloud.mapDataPath;
        string mapKey = mapDataCloud.key;
        bool isDownloaded = processMapDataDisplay.IsDownloaded(mapKey);
        
        // If the map is not downloaded do it
        if (!isDownloaded)
        {
            cloudDownload.TryDownloadMap(mapPath, mapKey, OnDownloadSuccess, OnDownloadError);
            Debug.Log("Map downloaded successfully");
            processMapDataDisplay.LoadMaps();
        }

        // Update the listing info and the map data
        localIndexSave.UpdateListingInfo(saveType);
        UpdateMapData();

        // After ensuring the map is downloaded and updated, enter the game mode
        if (isDownloaded)
        {
            EnterGameMode();
        }
    }

    public void PullMap()
    {
        MapData_Cloud mapDataCloud = processMapDataDisplay.GetCurrentMap();
        string mapKey = mapDataCloud.key;
        string lastMod = mapDataCloud.lastModifiedDate;
        warning.ShowWarning("Checking if the map is up to date...");

        string cooldownTime = cooldown.TryUpdateInteraction(mapKey, CRUD_EasySave3_Cooldown.MapInteractionType.Pull);
        if (cooldownTime != "-1s")
        {
            warning.ShowWarning($"You need to wait {cooldownTime} before you can do that again.");
            return;
        }

        // Check if the map is up to date
        cloudDownload.IsMapUpToDate(mapKey, lastMod, (updatedMap, updateNeeded) =>
        {
            // Map needs to be updated, populate the interaction menu
            PopoulateInteractionMenu(updatedMap);
            processMapDataDisplay.SetCurrentMap(updatedMap);
            processMapDataDisplay.LoadMaps();
            localIndexSave.UpdateListingInfo(saveType);
            if (updateNeeded)
            {
                warning.ShowWarning("Updating map data...");
                Debug.Log("Map needs to be updated");
                // Use the updated map data
                string mapPath = updatedMap.mapDataPath;
                string mapKey = updatedMap.key;
                bool isDownloaded = processMapDataDisplay.IsDownloaded(mapKey);

                cloudDownload.TryDownloadMap(mapPath, mapKey, OnDownloadSuccess, OnDownloadError);
                Debug.Log("Map downloaded successfully");
                processMapDataDisplay.LoadMaps();
            }
            else
            {
                Debug.Log("Map data updated!");
                warning.ShowWarning("Map data updated!", Warning.WarningType.Positive);
            }

            // Update the listing info and map data
            localIndexSave.UpdateListingInfo(saveType);
            UpdateMapData();            
        });
    }

    // If download is successful
    private void OnDownloadSuccess()
    {
        processMapDataDisplay.UpdateDownloadedMapsDict();
        localIndexSave.UpdateListingInfo(saveType);
        UpdateMapData();
        uiMapDataDisplay.SetInteractionButton(true);
        warning.ShowWarning("Map downloaded successfully.", Warning.WarningType.Positive);
        Debug.Log("Map downloaded successfully.");
        
        // If the map is downloaded, increment the download count, we count 1 download per day for the same map to avoid spam as it only increments and doesn't have account check <- easy solution instead of saving anonymous user data etc etc... too complicated
        string mapKey = processMapDataDisplay.GetMapKey();
        string cooldownTime = cooldown.TryUpdateInteraction(mapKey, CRUD_EasySave3_Cooldown.MapInteractionType.Download);
        if (cooldownTime != "-1s")
        {
            Debug.Log($"You need to wait {cooldownTime} before you can do that again.");
            return;
        }
        userDataManager.IncrementMapDownloadCount(mapKey, (success) =>
        {
            if (success)
            {
                Debug.Log("Count updated successfully.");
            }
            else
            {
                Debug.LogError("Failed to update count.");
            }
        });
    }

    // Error handling
    private void OnDownloadError(Exception e)
    {
        warning.ShowWarning("Failed to download the map.", Warning.WarningType.Negative);
        Debug.LogError("Failed to download the map:" + e);
    }

    // Update map data (name and description) -> U
    public bool UpdateMapData()
    {   
        Debug.Log("Updating map data...");
        string mapKey = processMapDataDisplay.GetMapKey();
        bool isDownloaded = processMapDataDisplay.IsDownloaded(mapKey);
        uiMapDataDisplay.SetInteractionButton(isDownloaded);

        return isDownloaded;
    }

    public void TryUpdateMap()
    {
        UpdateMapData();
    }

    // Delete the current map -> D
    public void DeleteMap()
    {
        MapData_Cloud currentMap = processMapDataDisplay.GetCurrentMap();
        string mapKey = processMapDataDisplay.GetMapKey();
        if (currentMap == null || !processMapDataDisplay.IsDownloaded(mapKey))
        {
            Debug.Log("Map not found");
            return;
        }

        // Delete the map listing info
        processMapDataDisplay.RemoveMap(mapKey);
        localIndexSave.UpdateListingInfo(saveType);

        // Delete the map data 
        tilesDataSave.DeleteMap(mapKey, saveType);

        // Reload the maps and clear the current map
        processMapDataDisplay.LoadMaps();
        processMapDataDisplay.SetCurrentMap(null);

        // Close the menu that interacts with the map after deleting it 
        uiMapDataDisplay.SetInteractionButton(false);
        uiMapDataDisplay.SetInteractionMenuState(false);
        Debug.Log("Map deleted successfully");

        warning.ShowWarning("Map deleted successfully", Warning.WarningType.Positive);
    }

    // Cast a vote for the map
    public void VoteMap()
    {
        MapData_Cloud currentMap = processMapDataDisplay.GetCurrentMap();
        string mapKey = processMapDataDisplay.GetMapKey();
        if (currentMap == null || !processMapDataDisplay.IsDownloaded(mapKey))
        {
            Debug.Log("Map not found");
            return;
        }

        int voteValue = uiMapDataDisplay.GetVoteValue();

        userDataManager.VoteMap(currentMap.key, voteValue, (success) =>
        {
            if (success)
            {
                Debug.Log("Vote submitted successfully.");
                warning.ShowWarning("Vote submitted successfully", Warning.WarningType.Positive);
                // Update the map data with the new vote value
                currentMap.likes += voteValue; // It is locally stored, if updated it would pull the data from the cloud
                PopoulateInteractionMenu(currentMap);
                processMapDataDisplay.SetCurrentMap(currentMap);
                processMapDataDisplay.LoadMaps();
                localIndexSave.UpdateListingInfo(saveType);
            }
            else
            {
                Debug.LogError("Failed to submit vote.");
                warning.ShowWarning("Failed to submit vote", Warning.WarningType.Negative);
            }
        });
    }

    // Cast a rating for the map
    public void RateMap()
    {
        MapData_Cloud currentMap = processMapDataDisplay.GetCurrentMap();
        string mapKey = processMapDataDisplay.GetMapKey();
        if (currentMap == null || !processMapDataDisplay.IsDownloaded(mapKey))
        {
            Debug.Log("Map not found");
            return;
        }

        int rateValue = uiMapDataDisplay.GetRateValue();

        userDataManager.RateMap(currentMap.key, rateValue, (success, rateCountDelta) =>
        {
            if (success)
            {
                Debug.Log("Rating submitted successfully.");
                warning.ShowWarning("Rating submitted successfully", Warning.WarningType.Positive);
                // Update the map data with the new rating value
                currentMap.totalRating += rateValue; // It is locally stored, if updated it would pull the data from the cloud
                currentMap.rates += rateCountDelta; // Increment the number of rates if the user did not rate the map before
                PopoulateInteractionMenu(currentMap);
                processMapDataDisplay.SetCurrentMap(currentMap);
                processMapDataDisplay.LoadMaps();
                localIndexSave.UpdateListingInfo(saveType);
            }
            else
            {
                Debug.LogError("Failed to submit rating.");
                warning.ShowWarning("Failed to submit rating", Warning.WarningType.Negative);
            }
        });
    }




}
