using UnityEngine;
using TMPro;
using UnityEngine.UI;
using System.Collections.Generic;
using System.Linq;
using System;
using System.Collections;

// Handles the interaction with the map data -> TODO error handling
public class Manager_MapDataLocal : MonoBehaviour
{
    [SerializeField] private Interact_MapDataDisplayLocal processMapDataDisplay; // Reference to the info / processing of the display of the map data
    [SerializeField] private Manager_MapAccess mapAccess; // Reference to the map access manager script
    [SerializeField] private UI_MapDataDisplay uiMapDataDisplay; // Reference to the map menus manager script
    [SerializeField] private Manager_IndexData localIndexSave; // Reference to the local save script
    [SerializeField] private Manager_TilesData tilesDataSave; // Reference to the map tiles manager script
    [SerializeField] private Manager_SceneTransitor sceneTransitorManager; // Reference to the scene transitor manager script
    [SerializeField] private Button stopSharingButton; // Reference to the stop sharing button
    public Warning warning; // Reference to the warning script
    // Data origin
    private LocalSaveKeys.SaveType saveType = LocalSaveKeys.SaveType.Local; // The save type of the map data

    private void Awake()
    {
        // Load the existing maps and display them
        processMapDataDisplay.SetMaps(localIndexSave.GetMapListingInfo(saveType));
        processMapDataDisplay.LoadMaps();

        // Gets listeners for the input fields
        uiMapDataDisplay.GetInputFields(out var editTitleName, out var editDescription);
        editTitleName.onEndEdit.AddListener(delegate { UpdateMapData(); });
        editDescription.onEndEdit.AddListener(delegate { UpdateMapData(); });
    }

    /***** UI interaction ****/

    // Load the map for playing
    public void EnterGameMode()
    {
        UpdateMapData();
        Debug.Log("Loading map for play: " + processMapDataDisplay.GetMapKey());
        sceneTransitorManager.LoadLevelEditorPlay();
    }

    // Edit the map
    public void EnterEditMode()
    {
        UpdateMapData();
        Debug.Log("Loading map editing scene: " + processMapDataDisplay.GetMapKey());
        sceneTransitorManager.LoadLevelEditor();
    }

    // Add data to the interaction menu
    public void PopoulateInteractionMenu(MapData_ES3 mapData)
    {
        uiMapDataDisplay.ShowMapEdit(mapData);
        processMapDataDisplay.SetCurrentMap(mapData);
        if (mapData.isUploaded)
        {
            stopSharingButton.interactable = true;
        }
        else
        {
            stopSharingButton.interactable = false;
        }
    }

    // Check if the map is verified
    public bool IsMapVerified()
    {
        return processMapDataDisplay.GetCurrentMap().isValidated;
    }

    /**** CRUD operations ****/

    // Create a new map -> C
    public void CreateMap()
    {
        string name = processMapDataDisplay.GetPlaceholderName();

        if (processMapDataDisplay.IsNameUnique(name))
        {
            string mapKey = processMapDataDisplay.NewMap(name);
            localIndexSave.UpdateListingInfo(saveType);
            Debug.Log("Map created successfully");

            // Enter the map editing mode
            Debug.Log("Loading map editing scene: " + name);
            PlayerPrefs.SetString(KeyManager.saveKeysDict[KeyManager.SaveKey.EditMapKey], mapKey);
            sceneTransitorManager.LoadLevelEditor();
        }
        else
        {
            Debug.Log("Map already exists, change name and try again");
            warning.ShowWarning("Map already exists, change name and try again", Warning.WarningType.Negative);
        }        
    }

    // Create a duplicate of the current map -> C
    public void DuplicateMap()
    {
        // Check if the map is selected
        MapData_ES3 currentMap = processMapDataDisplay.GetCurrentMap();
        if (currentMap == null || !processMapDataDisplay.ContainsMap(currentMap.key))
        {
            Debug.Log("Map not found");
            return;
        }

        // Get the new map name and paths to make the duplication
        string duppName = processMapDataDisplay.GetNotDuplicatedName(currentMap.name);

        if (processMapDataDisplay.IsNameUnique(duppName))
        {
            string newMapKey = processMapDataDisplay.NewMap(duppName, currentMap.description);
            localIndexSave.CreateCopyOfMap(currentMap.key, newMapKey);
            Debug.Log("Map duplicated successfully");
            uiMapDataDisplay.SetInteractionMenuStatus(false);
        }
        else
        {
            Debug.Log("Map already exists, change name and try again");
        }

        // Update the listing info and clear the current map
        localIndexSave.UpdateListingInfo(saveType);
        processMapDataDisplay.LoadMaps();
        processMapDataDisplay.SetCurrentMap(null);
        warning.ShowWarning("Map duplicated successfully", Warning.WarningType.Positive);
    }

    // Tries to get the map details and returns true if successful -> R
    public MapData_ES3 GetMapDetails()
    {
        return processMapDataDisplay.GetCurrentMap();
    }

    // Get the map tiles data -> R -> TODO we may have error to read big datafiles
    public string GetMapTilesData(string mapKey)
    {
        return tilesDataSave.GetMapData(mapKey);
    }

    // Update map data (name and description) -> U
    public bool UpdateMapData()
    {
        bool didChange = false;
        MapData_ES3 currentMap = processMapDataDisplay.GetCurrentMap();
        // If no map is selected, return
        if (currentMap == null)
        {
            Debug.Log("No map selected");
            return false;
        }

        // Get the map details and check that we are updating the correct map
        uiMapDataDisplay.GetInputFields(out var editTitleName, out var editDescription);
        string newName = editTitleName.text;
        string newDescription = editDescription.text;

        // Given a created map, check if the name is already taken
        if (!processMapDataDisplay.IsNameUnique(newName) && newName != currentMap.name)
        {
            editTitleName.text = currentMap.name;
            Debug.Log("Map name already taken, change name and try again");
            warning.ShowWarning("Map name already taken, change name and try again", Warning.WarningType.Negative);
            return false;
        }

        // The new map name is different from the current map name and it is available
        if (newName != currentMap.name)
        {
            // If the map is uploaded, we cannot rename it
            if (currentMap.isUploaded)
            {
                Debug.Log("Map is uploaded, cannot rename");
                editTitleName.text = currentMap.name;
            }
            else
            {
                processMapDataDisplay.UpdateMapName(newName);
                didChange = true;
            }
        }

        // Update the map description
        if (newDescription != currentMap.description)
        {
            processMapDataDisplay.UpdateMapDescription(newDescription);
            didChange = true;
        }

        // Save the changes
        localIndexSave.UpdateListingInfo(saveType);
        processMapDataDisplay.LoadMaps();
        Debug.Log("Map data updated!");
        if (didChange)
        {
            warning.ShowWarning("Map data updated successfully", Warning.WarningType.Positive);
        }

        return true;
    }

    // Update upload status of the current map -> U
    public void UpdateUploadStatus(bool isUploaded)
    {
        Debug.Log("Updating upload status of the map");
        if (processMapDataDisplay.UpdateUploadStatus(isUploaded))
        {
            localIndexSave.UpdateListingInfo(saveType);
            uiMapDataDisplay.UpdateTitleState(isUploaded);
            processMapDataDisplay.LoadMaps();
            Debug.Log("Map upload status saved");
        }
        else
        {
            Debug.Log("Map not found");
        }
    }

    // Delete the current map -> D
    public void DeleteMap()
    {
        MapData_ES3 currentMap = processMapDataDisplay.GetCurrentMap();
        
        if (currentMap == null || !processMapDataDisplay.ContainsMap(currentMap.key))
        {
            Debug.Log("Map not found");
            return;
        }

        // Delete the map listing ifno
        processMapDataDisplay.RemoveMap(currentMap.key);
        localIndexSave.UpdateListingInfo(saveType);

        // Delete the map data 
        tilesDataSave.DeleteMap(currentMap.key, saveType);

        // Reload the maps and clear the current map
        processMapDataDisplay.LoadMaps();
        processMapDataDisplay.SetCurrentMap(null);

        // Close the menu that interacts with the map after deleting it 
        uiMapDataDisplay.SetInteractionMenuStatus(false);
        Debug.Log("Map deleted successfully");

        warning.ShowWarning("Map deleted successfully", Warning.WarningType.Positive);
    }

    // Stop sharing the map
    public void StopSharingMap()
    {
        MapData_ES3 currentMap = processMapDataDisplay.GetCurrentMap();
        if (currentMap == null)
        {
            Debug.Log("Map not found");
            return;
        }

        mapAccess.TryDeleteMap(currentMap.key, success =>
        {
            if (!success)
            {
                warning.ShowWarning("Failed to stop sharing map", Warning.WarningType.Negative);
                return;
            }
            //UnityMainThreadDispatcher.Instance.Enqueue(() => {  });
            stopSharingButton.interactable = false;
            
            currentMap.isUploaded = false;
            UpdateUploadStatus(false);
            PopoulateInteractionMenu(currentMap);            
            warning.ShowWarning("Stopped sharing the map successfully", Warning.WarningType.Positive);
        });
    }
}
