using UnityEngine;
using System.Collections.Generic;
using System.Linq;
using System.Globalization;
using System;

public class Interact_MapDataDisplayLocal : Interact_MapDisplayBase
{
    private MapData_ES3 currentMap = null;
    private bool isCloud = false;

    // Instantiate a local map display
    protected override void InstantiateMapDisplay(IMapData map)
    {
        MapData_ES3 mapData = map as MapData_ES3;
        if (mapData != null)
        {
            GameObject newMapDisplay = Instantiate(mapDisplayPrefab, mapsViewport);
            newMapDisplay.GetComponent<Object_MapDisplay>().SetMapData(mapData);
        }
    }
    
    /****** Specific Operations ******/

    // Check if a map with the given name exists, return true if it does
    public bool ContainsMap(string name)
    {
        return maps.ContainsKey(name);
    }

    // Check if the name is unique, and thus can be used for a new map
    public bool IsNameUnique(string name)
    {
        foreach (var map in maps.Values)
        {
            if (map.Name == name)
            {
                return false; // Found a duplicate name
            }
        }
        return true; // Name is unique
    }

    // Check if the current map is uploaded
    public bool IsMapUploaded()
    {
        return currentMap != null && currentMap.isUploaded;
    }

    // Add a new map to the dictionary
    public void AddNewMap(MapData_ES3 newMap)
    {
        maps.Add(newMap.key, newMap);
    }

    // Remove a map from the dictionary
    public void RemoveMap(string name)
    {
        maps.Remove(name);
    }

    // Create a new map
    public string NewMap(string name, string description)
    {
        MapData_ES3 newMap = new MapData_ES3(name) { description = description }; // Create new map
        AddNewMap(newMap);
        SetCurrentMap(newMap);
        InstantiateMapDisplay(newMap);
        return newMap.key;
    }

    // Creates a new map with only a name
    public string NewMap(string name)
    {
        string mapKeyName = NewMap(name, "Map description...");
        Debug.Log("Map created successfully: " + mapKeyName);
        return mapKeyName;
    }

    // Set the current map
    public void SetCurrentMap(MapData_ES3 map)
    {
        currentMap = map;
    }

    // Update name of a map
    public void UpdateMapName(string newName)
    {
        currentMap.name = newName;
        Debug.Log("Map name updated successfully: " + newName);
    }

    // Update description of a map
    public void UpdateMapDescription(string description)
    {
        currentMap.description = description;
        Debug.Log("Map description updated successfully: " + description);
    }

    // Update upload status of a map
    public bool UpdateUploadStatus(bool isUploaded)
    {
        if (currentMap == null) return false; // Check if there is a current map

        MapData_ES3 map = (MapData_ES3) maps[currentMap.key];
        map.isUploaded = isUploaded;
        Debug.Log("Map upload status updated to: " + isUploaded);
        
        return true;
    }

    public void SetCloud(bool isCloud)
    {
        this.isCloud = isCloud;
    }

    // Load shared maps
    public void LoadSharedMaps()
    {
        if (maps == null) return;
        ClearMapsDisplay();

        // Order by newest first
        var sharedMaps = maps.Values
            .OfType<MapData_ES3>()
            .Where(map => map.isUploaded)
            .OrderBy(map => DateTime.ParseExact(map.lastModifiedDate, "dd-MM-yyyy HH:mm:ss", CultureInfo.InvariantCulture));

        foreach (var map in sharedMaps)
        {
            InstantiateMapDisplay(map);
        }
    }

    // Override the LoadMaps method to load only shared maps
    public override void LoadMaps()
    {
        if (maps == null) return;
        if (isCloud)
        {
            LoadSharedMaps();
            return;
        }
        ClearMapsDisplay();

        // Order by newest first
        var sharedMaps = maps.Values
            .OfType<MapData_ES3>()
            .OrderBy(map => DateTime.ParseExact(map.lastModifiedDate, "dd-MM-yyyy HH:mm:ss", CultureInfo.InvariantCulture));

        foreach (var map in sharedMaps)
        {
            InstantiateMapDisplay(map);
        }
    }
    
    /****** Getters ******/

    // Get the name of the current map
    public string GetMapName()
    {
        return currentMap == null ? "Unnamed" : currentMap.name;
    }

    // Get the key of the current map
    public string GetMapKey()
    {
        return currentMap == null ? "No key" : currentMap.key;
    }

    // Get the description of the current map
    public string GetMapDescription()
    {
        return currentMap == null ? "Map description..." : currentMap.description;
    }

    // Generate a placeholder name for a new map
    public string GetPlaceholderName()
    {
        return "Unnamed map " + (maps.Count + 1);
    }

    // Get the current map
    public MapData_ES3 GetCurrentMap()
    {
        return currentMap;
    }

    // Get a name that is not duplicated
    public string GetNotDuplicatedName(string originalName)
    {
        string name = originalName;
        int i = 1;
        while (!IsNameUnique(name))
        {
            name = originalName + " (" + i + ")";
            i++;
        }

        return name;
    }
}
