using System.Collections.Generic;
using UnityEngine;
// Purpose: Contains the classes for storing map data.

// Interface for map data
public interface IMapData
{
    string Name { get; }
    string Description { get; }
}

// Attributes of map data that is used to display them in the UI
public class MapData_Cloud : IMapData
{
    public string key; // Unique key of the item -> UUID
    public string mapName;
    public string mapAuthor;
    public string mapAuthorID;
    public string mapDescription;
    public Dictionary<int, HoleInfo> holesPar = new Dictionary<int, HoleInfo>(); // Par of the holes
    public string lastModifiedDate; // Date of last modification
    public string creationDate; // Date of creation
    public string mapDataPath; // Path to the map data file
    // Additional attributes TODO
    public int downloads = 0; // Number of downloads
    public int likes = 0; // Number of likes
    public int totalRating = 0; // total
    public int rates = 0; // Number of ratings
    public string fileSize; // Size of the map file
    // Implement IMapData
    public string Name => mapName;
    public string Description => mapDescription;

    public override string ToString()
    {
        return $"Key: {key}, Name: {mapName}, Author: {mapAuthor}, AuthorID: {mapAuthorID}, " +
            $"Description: {mapDescription}, Created: {creationDate}, Modified: {lastModifiedDate}, " +
            $"Path: {mapDataPath}, Downloads: {downloads}, Likes: {likes}, " +
            $"Rating: {totalRating/rates} ({rates} rates), File Size: {fileSize}, Holes: {holesPar.Count}";
    }
}

public class HoleInfo
{
    public int parValue;
    public float posX;
    public float posY;

    public HoleInfo()
    {
        parValue = 1;
        posX = 0.5f;
        posY = -4.5f;
    }

    public HoleInfo(int parValue, float posX, float posY)
    {
        this.parValue = parValue;
        this.posX = posX;
        this.posY = posY;
    }

    public Vector3 GetPosition()
    {
        return new Vector3(posX, posY, 0);
    }
}

// Attributes of map data that is used to store them locally
public class MapData_ES3 : IMapData
{
    public string key; // Unique key of the item -> UUID
    public string name; // Name of the map
    public string description = "Description goes here..."; // Description of the map
    public bool isUploaded = false; // If the map has been uploaded
    public bool isValidated = false; // If the map has been validated
    public Dictionary<int, HoleInfo> holesPar = new Dictionary<int, HoleInfo>(); // Par of the holes
    public string creationDate; // Date of creation
    public string lastModifiedDate; // Date of last modification
    public string fileSize; // Size of the map file

    public MapData_ES3(string name)
    {
        this.name = name;
        this.key = UUIDGenerator.GetUniqueKey(); // Generate a unique key
        this.creationDate = DateTimeGenerator.GetCurrentDateTime(); // Set creation date
        this.lastModifiedDate = this.creationDate; // Set last modified date
        holesPar.Add(1, new HoleInfo());
    }

    // Implement IMapData
    public string Name => name;
    public string Description => description;

    // Static method to calculate total strokes (sum of all par values)
    public static int CalculateTotalStrokes(Dictionary<int, HoleInfo> holesPar)
    {
        int totalStrokes = 0;
        foreach (KeyValuePair<int, HoleInfo> pair in holesPar)
        {
            totalStrokes += pair.Value.parValue; // Add each hole's par value to the total
        }
        return totalStrokes;
    }

    // Static method to calculate the number of holes (count of entries in the dictionary)
    public static int CalculateNumberOfHoles(Dictionary<int, HoleInfo> holesPar)
    {
        return holesPar.Count; // Return the number of entries in the dictionary
    }

    // Calculate the average par value
    public static float GetAveragePar(Dictionary<int, HoleInfo> holesPar)
    {
        int totalStrokes = CalculateTotalStrokes(holesPar);
        int numberOfHoles = CalculateNumberOfHoles(holesPar);
        if (numberOfHoles == 0) return 0; // Avoid division by zero
        return (float)totalStrokes / numberOfHoles; // Return the average
    }
}