using UnityEngine;
using System;
using System.Collections;

public class Manager_Camera : MonoBehaviour
{
  [SerializeField] ObjectFollower objectFollower = null;
  [SerializeField] PanZoom panZoom = null;
  [SerializeField] private Rigidbody2D ballRigidbody;       // Rigidbody of the ball (player)
  
  [SerializeField] private float ballStoppedThreshold = 0.1f; // Velocity below which ball is considered stopped
  [SerializeField] private float resetDelay = 1.0f;           // Delay before returning to explore mode on reset

  // Flags
  [Header("Flags")]
  private bool isExploring = true;
  private bool isFollowing = false;
  private bool awaitingBallStop = false;

  // Save the couroutine so it doesnt call multiple times
  private Coroutine delayedEnableExploreCoroutine = null;
  void Update()
  {
    // After a shot, check if ball has stopped and no aiming in progress
    if (awaitingBallStop)
    {
      if (ballRigidbody.linearVelocity.sqrMagnitude <= ballStoppedThreshold)
      {
        // Ball has come to rest, switch to exploring mode
        awaitingBallStop = false;
        SetFollow(false);
        SetExploring(true);
        Debug.Log("Ball stopped, switching to explore mode.");
      }
    }
  }

  // Call when the player starts aiming or charging a shot.
  public void OnStartAiming()
  {
    // Immediately cancel exploring, switch to follow mode
    SetExploring(false);
    // (Optionally set a flag if we needed to track aiming state)
    TryStopCoroutine();
  }

  // Call when the player releases the shot (shoot action completed).
  public void OnShotFired()
  {
    // Ensure camera is following during the shot
    SetFollow(true);
    TryStopCoroutine();
    // Mark that we should watch the ball
    awaitingBallStop = true;
  }

  // Call when the user requests a view reset (e.g. via UI)
  public void ResetView()
  {
    Debug.Log("Resetting camera view to follow player.");
    // Snap camera to follow player
    SetFollow(true);
    SetExploring(false);
    panZoom.ResetView(); // Reset pan/zoom to default state
    awaitingBallStop = false; // Reset awaiting state
                              // After a delay, return to explore mode
    TryStopCoroutine();
    delayedEnableExploreCoroutine = StartCoroutine(DelayedEnableExplore());
  }

  private void TryStopCoroutine()
  {
    if (delayedEnableExploreCoroutine != null)
    {
      StopCoroutine(delayedEnableExploreCoroutine);
      delayedEnableExploreCoroutine = null; // Clear reference
      Debug.Log("Stopped previous delayed enable explore coroutine.");
    }
  }

  // Internal helper to return to explore mode after a delay
  private IEnumerator DelayedEnableExplore()
  {
    float time = GetTimeToTarget();
    Debug.Log($"Waiting {time} seconds before enabling explore mode.");
    yield return new WaitForSeconds(time + 0.05f);
    SetFollow(false);
    SetExploring(true);
    Debug.Log("Explore mode enabled after delay.");

    delayedEnableExploreCoroutine = null; // clear reference
  }

  // Enables or disables exploring mode (pan/zoom).
  // If true, enable exploring (disable follow). If false, disable exploring (enable follow).
  public void SetExploring(bool on)
  {
    isExploring = on;
    if (panZoom != null)
        panZoom.enabled = on;        // Enable pan/zoom when exploring
    if (objectFollower != null)
        objectFollower.follow = !on; // Disable follow when exploring

    // Optionally: update isFollowing state
    isFollowing = !on;
  }

  public void SetFollow(bool on)
  {
    isFollowing = on;
    if (panZoom != null)
        panZoom.enabled = !on;       // Disable pan when following
    if (objectFollower != null)
        objectFollower.follow = on;  // Enable follow when following

    // Optionally: update isExploring state
    isExploring = !on;
  }

  public float GetTimeToTarget()
  {
    return objectFollower.GetTimeToTarget();
  }

  public void SetPositionToTarget()
  {
    objectFollower.SetPositionToTarget();
    Debug.Log("Camera position set to target.");
  }

 



}