using UnityEngine;
using System.Collections.Generic;
using UnityEngine.UI;

public class Manager_Player : MonoBehaviour
{
    private Transform playerPos;
    private Rigidbody2D playerRb;
    public PlayerController playerController;
    private PlayerController.BallTrigger playerDying = PlayerController.BallTrigger.Die;
    private PlayerController.BallTrigger playerAppearing = PlayerController.BallTrigger.Appear;
    private bool firstLoad = true;
    [SerializeField] private List<Vector2> lastValidPositions = new List<Vector2>();
    [SerializeField] private Transform currentSpawnPoint;
    [SerializeField] private Manager_Camera cameraManager;
    [SerializeField] private Button returnToLastValidPosButton;
    
    public void Awake()
    {
        FindPlayer();
        returnToLastValidPosButton.interactable = false; // Disable the button initially
    }
    
    // Find the player object by tag and get the player data
    public void FindPlayer() 
    {
        // Find the player object by tag
        GameObject player = GameObject.FindWithTag("Player");

        // Get the player controller component
        playerController = player.GetComponent<PlayerController>();

        // Get the player's position and rigidbody
        playerPos = player.transform;
        playerRb = player.GetComponent<Rigidbody2D>();
    }
    
    // Set the player to a new position
    public void SetPlayerNewPos(Transform newPos)
    {
        playerRb.linearVelocity = Vector2.zero;
        // Assuming playerPos and currentSpawnPoint are Transform components
        Vector3 currentPos = playerPos.position; // Get the current player position

        // Set the x and y coordinates of the player's position to match the spawn point's x and y
        currentPos.x = newPos.position.x;
        currentPos.y = newPos.position.y;

        // Assign the modified position back to the player's position, retaining the z coordinate
        playerPos.position = currentPos;
    }

    // Set the player to a new position
    public void SetPlayerNewPos(Vector2 newPos)
    {
        playerRb.linearVelocity = Vector2.zero;
        // Assuming playerPos and currentSpawnPoint are Transform components
        Vector3 currentPos = playerPos.position; // Get the current player position

        // Set the x and y coordinates of the player's position to match the spawn point's x and y
        currentPos.x = newPos.x;
        currentPos.y = newPos.y;

        // Assign the modified position back to the player's position, retaining the z coordinate
        playerPos.position = currentPos;
    }

    //  Add new latest valid positions
    public void AddLastValidPosition(Vector2 newPosition)
    {
        lastValidPositions.Add(newPosition);
        returnToLastValidPosButton.interactable = true; // Enable the button when a valid position is added
    }

    // Return the last valid position
    public void ReturnToLastValidPosition()
    {
        if (lastValidPositions.Count > 0)
        {
            Vector2 lastValidPos = lastValidPositions[lastValidPositions.Count - 1];
            lastValidPositions.RemoveAt(lastValidPositions.Count - 1);
            SetPlayerNewPos(lastValidPos);
        }
        else
        {
            // If there are no valid positions, return the current spawn point
            ReturnToSpawnPoint();
        }

        returnToLastValidPosButton.interactable = lastValidPositions.Count > 0? true : false; // Disable the button if no valid positions are available
        cameraManager.ResetView();
    }

    // Return to the spawn point
    public void ReturnToSpawnPoint()
    {
        SetPlayerNewPos(currentSpawnPoint);
    }

    // Set the spawn point
    public void SetSpawnPoint(Transform spawnPoint)
    {
        currentSpawnPoint = spawnPoint;
    }

    // Load the level with a new spawn point
    public void LoadLevel(Transform newSpawnPoint)
    {
        currentSpawnPoint = newSpawnPoint;
        lastValidPositions =  new List<Vector2>(); // Reset the last valid positions as we are starting a new level
        returnToLastValidPosButton.interactable = false; // Disable the button when loading a new level
        SetPlayerNewPos(currentSpawnPoint);
    }

    // Stop the player and set the dying trigger
    public void StopPlayerAndDie()
    {
        playerRb.linearVelocity = Vector2.zero;
        playerController.SetDying(true);
        playerController.SetTrigger(playerDying);
    }

    public void MakePlayerAppear()
    {
        playerController.SetTrigger(playerAppearing);
        playerController.SetDying(false);
    }

    // Make the player appear and stop dying
    public void MakePlayerAppear(bool isEditorMode, bool isNewLoad = false)
    {
        if (!firstLoad && isEditorMode)
        {
            playerController.SetTrigger(playerAppearing);
        }
        else
        {
            firstLoad = false;
        }
        playerController.SetDying(false, isNewLoad);
    }

    public void PropelPlayer(Vector2 direction, float force)
    {
        playerRb.linearVelocity = (direction.normalized) * force;
    }

}