using UnityEngine;
using TMPro;
using System.Collections.Generic;
using System.Collections;
using System;

public class Manager_Score : MonoBehaviour
{
  [System.Serializable]
  public class Score
  {
    public int starOne = 0;
    public int starTwo = 0;
    public int starThree = 0;
    public float time = 0;

    public Score(int starOne, int starTwo, int starThree, float time)
    {
      this.starOne = starOne;
      this.starTwo = starTwo;
      this.starThree = starThree;
      this.time = time;
    }
      
  }

  [SerializeField]
  public Score scorePoints = null; // TODO use scriptable object to store the score points
                                    // TODO refactor as UI for the text
  public ScoreData scoreData; // Reference to the score data
  [SerializeField] private TextMeshProUGUI scoreText; // Reference current amount of strokes
  [SerializeField] private TextMeshProUGUI scoreNotification; // Reference to notification text after each hole
  [SerializeField] private TextMeshProUGUI holeInOneNotification; // Reference the par indicator
  [SerializeField] private TextMeshProUGUI scoreParIndicator; // Reference the par indicator
  [SerializeField] private Manager_Level managerLevel; // Reference to the manager to get the par
  [SerializeField] private TimeController timeController; // Reference to the CourseScore script
  [SerializeField] private CourseScore courseScore; // Reference to the CourseScore script
  [SerializeField] UI_ResultsMenu uiResultsMenu;
  [SerializeField] UI_LevelRegistry uiLevelRegistry;
  private int strokes = 0; // The number of strokes taken by the player
  private int par = 0; // The par for the current hole
  private int levelIndex = 0; // The index of the current level
  [SerializeField] private bool nonStarMode = false;
  public event Action<bool> OnStrokeChanged;

  [SerializeField] private bool isLevel = false; // Flag to indicate if it is a predefined level or not, so it should recall the puntuation

  private void Awake()
  {
    levelIndex = PlayerPrefs.GetInt("LevelIndex"); // Set the isDownloaded flag to 0
    ResetStrokes(); // Update the score notification
    SetStarRatings();
  }
  private void Start()
  {
    StartCoroutine(Initialize());
  }

  private IEnumerator Initialize()
  {
    yield return new WaitForEndOfFrame();
    
    UpdateParIndicator();
    SetHoleNumber(managerLevel.GetCurrentLevel() + 1); // Set the hole number
    UpdateLevelData(); // Update the level data
  }

  private void UpdateLevelData()
  {
    if (!isLevel) return;
    if (managerLevel == null)
    {
      managerLevel = FindObjectOfType<Manager_Level>(); // Find the manager level if not set
    }
    Debug.Log("Updating level data for level index: " + levelIndex);
    Debug.Log("Level name: " + scoreData.scorePoints[levelIndex].levelName);
    managerLevel.SetLevelName(scoreData.scorePoints[levelIndex].levelName); // Set the level name
  }

  public void ShowScoreAndUpdateHole(int holeNum, int totalHoles)
  {
    uiLevelRegistry.SetScoreTrigger(UI_LevelRegistry.ScoreTrigger.Show.ToString());
    uiLevelRegistry.SetHoleNumber(holeNum, managerLevel.GetTotalLevels());

    if (holeNum >= totalHoles)
    {
      uiLevelRegistry.SetScoreTrigger(UI_LevelRegistry.ScoreTrigger.HideAll.ToString());
    }
  }

  public void HideAll()
  {
    uiLevelRegistry.SetScoreTrigger(UI_LevelRegistry.ScoreTrigger.HideAll.ToString());
  }

  // Set the hole number in the UI
  public void SetHoleNumber(int holeNum)
  {
    uiLevelRegistry.SetHoleNumber(holeNum, managerLevel.GetTotalLevels()); // Set the hole number
  }

  // Update the hole number in the UI
  public void UpdateHoleNumber()
  {
    int holeNum = managerLevel.GetCurrentLevel() + 1; // Get the current hole number
    uiLevelRegistry.SetHoleNumber(holeNum, managerLevel.GetTotalLevels()); // Set the hole number
  }

  public int Getstrokes()
  {
    return strokes; // Get the number of strokes
  }

  public void ResetStrokes()
  {
    SetStrokes(0); // Reset the score
    UpdateScoreText(strokes); // Update the score notification
    int currentLevel = managerLevel.GetCurrentLevel(); // Get the current level
    int totalLevels = managerLevel.GetTotalLevels(); // Get the total levels
    if (currentLevel >= totalLevels) return; // If the current level is greater than the total levels, return)
    Debug.Log("Resetting strokes for hole: " + currentLevel);
    uiLevelRegistry.SetHoleNumberIndicator(managerLevel.GetCurrentLevel() + 1); // Set the hole number
    uiLevelRegistry.SetScoreTrigger(UI_LevelRegistry.ScoreTrigger.ShowCurrentHole.ToString()); // If the reset is called, that means that the player is starting a new hole, so show the current hole
  }

  public void AddStrokes(int strokes)
  {
    OnStrokeChanged?.Invoke(strokes > par);
    SetStrokes(this.strokes + strokes); // Add the number of strokes to the current score
    UpdateScoreText(this.strokes); // Update the score notification
  }

  public void SetStrokes(int strokes)
  {
    this.strokes = strokes; // Set the number of strokes
  }

  public void UpdateScoreNotification()
  {
    string score = courseScore.GetScoreText(strokes, par); // Determine the score text
    holeInOneNotification.text = strokes == 1? "Hole-In-One" : ""; // Set the hole in one notification
    scoreNotification.text = score; // Set the text in TextMeshPro
  }

  public void UpdateParIndicator()
  {
    par = managerLevel.GetLevelPar(managerLevel.GetCurrentLevel()); // Get the par for the next hole
    scoreParIndicator.text = "Par " + par.ToString(); // Set the par indicator text
  }

  private void UpdateScoreText(int strokes)
  {
    this.strokes = strokes; // Update the number of strokes
    scoreText.text = strokes.ToString(); // Set the text in TextMeshPro
  }

  public void SetHoleScore(int holeNum)
  {
    courseScore.SetHoleScore(holeNum, strokes); // Set the hole score
  }

  public float GetTotalScore()
  {
    int strokes = courseScore.GetTotalStroke(); // Get the total score
    float time = timeController.GetElapsedTime(); // Get the total time
    return courseScore.GetTotalScore(strokes, time); // Get the total score
  }

  // Get the total score for the given course
  public int GetTotalStroke()
  {
    return courseScore.GetTotalStroke(); // Get the total score
  }

  // Stop the timer
  public void StopTimer()
  {
    timeController.SetGameEnded(true); // Stop the timer
  }

  // Get the total time for the given course
  public float GetTotalTime()
  {
    return timeController.GetElapsedTime(); // Get the total time
  }

  public void SaveScore(string levelName)
  {
    float totalTime = GetTotalTime(); // Get the total time
    courseScore.SaveScore(levelName, totalTime); // Save the score
  }

  // Set the star ratings
  public void SetStarRatings()
  {
    if (nonStarMode) return; // Return if the score points are null
    scorePoints = new Score(
      scoreData.scorePoints[levelIndex].starOne,
      scoreData.scorePoints[levelIndex].starTwo,
      scoreData.scorePoints[levelIndex].starThree,
      scoreData.scorePoints[levelIndex].time
    ); // Get the score points from the scriptable object

    int[] ratings = new int[3] { scorePoints.starOne, scorePoints.starTwo, scorePoints.starThree };
    float time = scorePoints.time;
    uiResultsMenu.SetStarRatings(ratings, time); // Set the star ratings
  }

  public void EndGame(Manager_Level.Level[] levels)
  {
    CalculateStarAchieved(); // Calculate the star achieved
    SetResultsScore(levels); // Set the results score
    uiLevelRegistry.ShowLevelSummary(); // Show the level summary
  }

  // Set the results score
  private void SetResultsScore(Manager_Level.Level[] levels)
  {
    Dictionary<int, int> holeScores = courseScore.GetHoleScores(); // Get the hole scores

    int totalPar = 0; // Initialize the total par
    for (int i = 0; i < levels.Length && holeScores.Count != 0; i++)
    {
      // If the score was not saved due to for example, starting from level != 0 just ignore it.
      if (!holeScores.ContainsKey(i))
        continue;
      int strokes = holeScores[i]; // Get the strokes for the hole
      int par = levels[i].parValue; // Get the par for the hole
      totalPar += par; // Add the par to the total par
      string hole = "Nº" + (i + 1).ToString(); // Get the hole number
      uiResultsMenu.AddHoleResult(hole, par, strokes); // Add the hole result
    }

    uiResultsMenu.SetTotalStrokes(courseScore.GetTotalStroke()); // Set the total strokes
    uiResultsMenu.SetTotalPar(totalPar); // Set the total par
  }
  // Calculate the star achieved
  private void CalculateStarAchieved()
  {
    int totalStrokes = GetTotalStroke(); // Get the total strokes
    float totalTime = GetTotalTime(); // Get the total time
    int totalScore = courseScore.GetTotalScore(totalStrokes, totalTime); // Get the total score

    uiResultsMenu.SetFinalScore(totalScore); // Set the final score

    string time = timeController.FloatToMinutes(totalTime); // Convert the time to minutes
    uiResultsMenu.SetTotalTime(time); // Set the total time
    if (!nonStarMode) uiResultsMenu.SetStarAchieved(totalStrokes, totalTime); // Set the star achieved
  }
}
