using UnityEngine;
using UnityEngine.EventSystems;
using UnityEngine.InputSystem;
using UnityEngine.InputSystem.Interactions;
using UnityEngine.Tilemaps;
using TMPro;
using UnityEngine.UI;
using System;
using UnityEngine.Events;

using System.Collections.Generic;

//Based on https://www.youtube.com/watch?v=hx3ieBzVNg0&list=PLJBcv4t1EiSz-wA35-dWpcI98pNiyK6an
/*
[TODO]
Bugs:
    -> After pressing the rotate button, if we had selected "free placement" it places without pressing.
    -> [BUG-1] Not erasing properly the obstacles and therefore the history does not work either.
QoL:
    -> Move Ball "dragging"
    -> Can't put items in the terrain layer which collides with other items.
    -> Show item in hand to indicate that we are using x item by marking the item in the left.
    -> Hold to undo or redo constantly.
    -> Show when the user is in "exploring mode" or in "placing mode"
Refactor:
    -> [REF-1] Make the code simpler, more readable or optmize it.

*/
public class BuildingCreator : MonoBehaviour
{
    [SerializeField] Tilemap previewTilemap, floorMap, terrainMap, floorItemsMap;
    [SerializeField] GameObject rotationButton;
    [SerializeField] Manager_Level managerLevel;
    [SerializeField] GameObject editorMenu;
    [SerializeField] GameObject constructionMenu;
    // Private member variables
    Tilemap baseMap;
    TileBase tileBase;
    [SerializeField] BuildingInputs buildingInputs;
    BuildingHistory history;
    BuildingObjectBase selectedObj, prevObj;
    [SerializeField] Camera previewCamera;
    [SerializeField] GameObject prevAndDelete_Left;
    [SerializeField] GameObject prevAndDelete_Right;
    Vector2 mousePosition;
    Vector3Int previousTilePosition;
    Vector3Int currentTilePosition;
    Vector3Int holdStartTilePosition;
    public event Action<bool> OnObjectRotated;

    bool placed;
    bool isDrawing;
    bool pressedOnUI;   

    public enum PlaceMode
    {
        Single,
        Free,
        Line,
        Area
    }
    PlaceMode placeMode;
    public TMP_Dropdown dropdown;

    void Start()
    {
        // Set default dropdown value
        dropdown.value = (int)placeMode;
        dropdown.onValueChanged.AddListener(SetPlaceModeFromDropdown);
    }

    // Iterates circularly trough the PlaceMode elements
    public void CircularPlaceModeSwitch()
    {
        placeMode = placeMode.Next();
    }

    // Pass the dropdown value directly to set the mode
    public void SetPlaceModeFromDropdown(int value)
    {
        placeMode = (PlaceMode)value;
    }

    protected void Awake()
    {
        buildingInputs = new BuildingInputs();
        history = GetComponent<BuildingHistory>();

        // Check if the input action asset is loaded correctly
        if (buildingInputs == null)
        {
            Debug.LogError("BuildingInputs not initialized correctly");
        }
    }

    // Register callbacks for position and click inputs
    private void OnEnable()
    {
        buildingInputs.Enable();
        buildingInputs.Building.Position.performed += OnMouseMove;
        buildingInputs.Building.Click.performed += OnClick;
        buildingInputs.Building.Click.started += OnClick;
        buildingInputs.Building.Click.canceled += OnClick;
    }

    // Unregister callbacks when disabled
    private void OnDisable()
    {
        buildingInputs.Building.Position.performed -= OnMouseMove;
        buildingInputs.Building.Click.performed -= OnClick;
        buildingInputs.Building.Click.started -= OnClick;
        buildingInputs.Building.Click.canceled -= OnClick;
        buildingInputs.Disable();
    }

    // Update mouse position when it changes
    private void OnMouseMove(InputAction.CallbackContext context)
    {
        mousePosition = context.ReadValue<Vector2>();
    }

    private bool IsPointerOverUIManual()
    {
        // Check if the pointer is over a UI element
        PointerEventData pointerData;

        if (Input.touchCount > 0)
        {
            Touch touch = Input.GetTouch(0);
            pointerData = new PointerEventData(EventSystem.current)
            {
                position = touch.position
            };
        }
        else
        {
            pointerData = new PointerEventData(EventSystem.current)
            {
                position = Input.mousePosition
            };
        }

        List<RaycastResult> results = new List<RaycastResult>();
        EventSystem.current.RaycastAll(pointerData, results);

        foreach (var result in results)
        {
            var gameObject = result.gameObject;
            // Check if the gameObject is active in the hierarchy
            if (!gameObject.activeInHierarchy)
                continue; // Skip inactive objects

            var graphic = gameObject.GetComponent<UnityEngine.UI.Graphic>();
            if (graphic != null && graphic.raycastTarget)
            {
                // Blocking UI detected
                return true;
            }
        }

        // No blocking UI found
        return false;
    }

    // Handle click events
    private void OnClick(InputAction.CallbackContext context)
    {
        // Don't do anything if the selected object is null
        if (selectedObj == null)
            return;
        
        // Use the manual UI check instead of EventSystem's cached one
        if (IsPointerOverUIManual() && selectedObj != null && !isDrawing)
        {
            //Debug.Log("Pointer is over UI");
            return;
        }

        // Handle the start and end of a click interaction
        if (context.phase == InputActionPhase.Started)
        {
            //Debug.Log("Click started");
            // Check for long tap interaction to initiate drawing
            if (context.interaction is SlowTapInteraction)
                isDrawing = true;
            // Store the initial tile position when drawing starts
            holdStartTilePosition = currentTilePosition;
            SetToggleActive(false);
        }
        else if (isDrawing && context.phase == InputActionPhase.Performed)
        {
            // End drawing when click is released
            isDrawing = false;
            HandleDrawingRelease();
            SetToggleActive(true);
            //Debug.Log("Click performed");
        }
    }

    // Hides or shows the UI elements based on the drawing state
    private void SetToggleActive(bool active)
    {
        editorMenu.SetActive(active);
        constructionMenu.SetActive(!active);
    }

    // Clear preview tiles and draw based on drawing mode
    private void HandleDrawingRelease()
    {
        previewTilemap.ClearAllTiles();
        if (ToolAction(baseMap))
            return;
        // Check if the user is canceling the action
        if (IsCancelingPlacement())
            return;

        switch (placeMode)
        {
            case PlaceMode.Single:
                DrawItem(baseMap);
                break;
            case PlaceMode.Line:
                DrawLine(baseMap);
                break;
            case PlaceMode.Area:
                DrawArea(baseMap);
                break;
        }
    }

    // Check if the user is canceling the placement
    private bool IsCancelingPlacement()
    {
       // Check if the pointer is over a UI element
        PointerEventData pointerData;

        if (Input.touchCount > 0)
        {
            Touch touch = Input.GetTouch(0);
            pointerData = new PointerEventData(EventSystem.current)
            {
                position = touch.position
            };
        }
        else
        {
            pointerData = new PointerEventData(EventSystem.current)
            {
                position = Input.mousePosition
            };
        }

        List<RaycastResult> results = new List<RaycastResult>();
        EventSystem.current.RaycastAll(pointerData, results);

        foreach (var result in results)
        {
            if (result.gameObject.CompareTag("CancelBuilding"))
            {
                return true;
            }
        }

        return false;
    }

    // Determine which tilemap to use based on current tile position
    // Usage: Get the tilemap from top floor to bottom floor
    private Tilemap GetTileMap()
    {
        return GetTileMap(currentTilePosition);
    }

    private Tilemap GetTileMap(Vector3Int pos)
    {
        if (selectedObj == null)
        {
            //Debug.LogError("Selected object is null");
            return null;
        }

        //Debug.Log("Selected object: " + selectedObj.name);
        //Debug.Log("Selected object category: " + selectedObj.Category);
        // A tool removes objects from top to bottom
        if (selectedObj.Category == Category.Tool)
        {
            //Debug.Log("Tool selected");
            TileBase tile = null;
            // Top level: Terrain
            tile = terrainMap.GetTile(pos);
            if (tile != null) return terrainMap;

            // Second level: Floor Items
            tile = floorItemsMap.GetTile(pos);
            if (tile != null) return floorItemsMap;

            // Third level: Floor
            tile = floorMap.GetTile(pos);
            if (tile != null) return floorMap;
        }
        return baseMap; // Return the base map based on the selected object category
    }

    // Obtains the current TileBase from the selected Object
    private TileBase GetTile()
    {
        TileBase tempTile;
        if (selectedObj != null && selectedObj.ToolType == ToolType.Eraser) tempTile = null;
        else tempTile = selectedObj.TileBase;
        return tempTile;
    }

    // Perform the action based on the selected tool if the tilemap is not null
    private bool ToolAction(Tilemap tilemap)
    {
        // Notice that we use the get function since we want to remove the top item
        // [TODO] refactor code to make it more comprehensible
        bool toolAction = false;
        Tilemap tMap = GetTileMap();
        
        if (tilemap == null)
        {
            switch (placeMode)
            {
                default:
                case PlaceMode.Free:
                    DrawItem(tMap);
                    break;
                case PlaceMode.Line:
                    DrawLine(tMap);
                    break;
                case PlaceMode.Area:
                    DrawArea(tMap);
                    break;
            }
            toolAction = true;          
        }
        return toolAction;
    }

    // Draw a single item on the tilemap while registering it to the history
    private void DrawItem(Tilemap tMap)
    {   
        tMap = GetTileMap(currentTilePosition);
        // The history is registered before since if we do it after, then the tile tMap.GetTile(currentTilePosition) will differ!
        if (tMap == null)
        {
            return;
        }

        history.AddItem(new BuildingHistoryStep(tMap, currentTilePosition, tMap.GetTile(currentTilePosition), GetTile()));
        SmartSetTile(tMap, currentTilePosition, GetTile());
    }

    // Draw a line using the item on the tilemap from start to current tile position
    private void DrawLine(Tilemap tMap)
    {
        // [REF-1] Refactor code to make it shorter and fix the eraser problem:
        // [BUG-1] When erasing, not doing it correctly with obstacles and ground tiles.
        float diffX = Mathf.Abs(holdStartTilePosition.x - currentTilePosition.x);
        float diffY = Mathf.Abs(holdStartTilePosition.y - currentTilePosition.y);

        bool isHorizontal = diffX > diffY;
        bool isTool = selectedObj.ToolType == ToolType.Eraser;

        
        int x = holdStartTilePosition.x;
        int y = holdStartTilePosition.y;

        int increment;

        List<Vector3Int> positions = new List<Vector3Int>();
        List<TileBase> prevTiles = new List<TileBase>();
        List<Tilemap> tileMaps = new List<Tilemap>();

        if (isHorizontal)
        {
            increment = (int)Mathf.Sign(currentTilePosition.x - holdStartTilePosition.x);
            
            do
            {
                Vector3Int pos = new Vector3Int(x, y, 0);
                positions.Add(pos);                
                tMap = GetTileMap(pos);
                tileMaps.Add(tMap);
                prevTiles.Add(tMap.GetTile(pos));
                x += increment;
            }
            while (x - increment != currentTilePosition.x);
        }
        else
        {
            increment = (int)Mathf.Sign(currentTilePosition.y - holdStartTilePosition.y);
            do
            {
                Vector3Int pos = new Vector3Int(x, y, 0);
                positions.Add(pos);                
                tMap = GetTileMap(pos);
                tileMaps.Add(tMap); 
                prevTiles.Add(tMap.GetTile(pos));

                y += increment;
            }
            while (y - increment != currentTilePosition.y);
        }

        history.AddItem(new BuildingHistoryStep(tileMaps.ToArray(), positions.ToArray(), prevTiles.ToArray(), GetTile()));
        
        // Now set the tiles after collecting history
        foreach (Vector3Int pos in positions)
        {
            SmartSetTile(GetTileMap(pos), pos, GetTile());
        }
    }

    // Draw an area using the item on the tilemap between start and current tile positions
    private void DrawArea(Tilemap tMap)
    {
        Vector3Int min = Vector3Int.Min(holdStartTilePosition, currentTilePosition);
        Vector3Int max = Vector3Int.Max(holdStartTilePosition, currentTilePosition);

        List<Vector3Int> positions = new List<Vector3Int>();
        List<TileBase> prevTiles = new List<TileBase>();
        List<Tilemap> tileMaps = new List<Tilemap>();

        for (int x = min.x; x <= max.x; x++)
        {
            for (int y = min.y; y <= max.y; y++)
            {
                Vector3Int pos = new Vector3Int(x, y, 0);
                tMap = GetTileMap(pos);
                tileMaps.Add(tMap);
                positions.Add(pos);
                prevTiles.Add(tMap.GetTile(pos));
            }
        }

        history.AddItem(new BuildingHistoryStep(tileMaps.ToArray(), positions.ToArray(), prevTiles.ToArray(), GetTile()));

        foreach (Vector3Int pos in positions)
        {
            SmartSetTile(GetTileMap(pos), pos, GetTile());
        }
    }

    private bool IsSpawnPoint(Tilemap tMap, Vector3Int pos)
    {
       TileBase tile = tMap.GetTile(pos);
       if (!tile) return false;
       Debug.Log(tile.name);
       return tile.name == "PosStart";
    }
    private void SmartSetTile(Tilemap tMap, Vector3Int pos, TileBase tile)
    {
        bool isEraser = selectedObj.ToolType == ToolType.Eraser;
        if (isEraser) {
            int totalLevels = managerLevel.GetTotalLevels();
            if (totalLevels == 1 && IsSpawnPoint(tMap, pos)) {
                Debug.Log("Can't erase the only spawn point");
                return;
            }
        }
        tMap.SetTile(pos, tile);
    }

    // Select a building object and update related properties
    public void ObjectSelected(BuildingObjectBase buildingObject)
    {
        selectedObj = buildingObject;
        UpdateTileBase();
        UpdateBaseMap();
        HasRotation();
        UpdatePreviewTilemap();
    }

    // Updates the tile base using the selectedObj
    private void UpdateTileBase()
    {
        // Evaluates that if selectedObj is not null, then acces the "TileBase" property 
        tileBase = selectedObj?.TileBase;
    }

    // Updates the tileMap we are drawing using the selectedObj
    private void UpdateBaseMap()
    {
        if (selectedObj == null)
        {
            baseMap = null;
            return;
        }

        switch (selectedObj.Category)
        {
            case Category.Terrain:
                baseMap = terrainMap;
                break;
            case Category.Floor:
                baseMap = floorMap;
                break;
            case Category.Floor_Items:
                baseMap = floorItemsMap;
                break;
            default:
                baseMap = floorItemsMap;
                break;
        }

        Debug.Log("BaseMap: " + baseMap);
    }

    // Updates the UI for rotating the object based on the attribute of the selectedObj
    private void HasRotation()
    {
        if (selectedObj != null && selectedObj.hasRotation) rotationButton.SetActive(true);
        else rotationButton.SetActive(false);
    }

    // Public method to rotate the selected object
    public void RotateObject()
    {
        if (selectedObj == null) return;
        selectedObj.UpdateIndex();
        // Trigger and ev3ent so the UI can update
        OnObjectRotated?.Invoke(true);
        tileBase = selectedObj.GetTile();
    }
    
    // Update function of unity
    private void Update()
    {
        if (selectedObj == null) return;

        // Convert mouse position to tile coordinates
        Vector3 mouseWorldPosition = Camera.main.ScreenToWorldPoint(mousePosition);

        // Make the preview camera follow the mouse position
        if (previewCamera != null)
        {
            Vector3 previewCameraPosition = new Vector3(mouseWorldPosition.x, mouseWorldPosition.y, previewCamera.transform.position.z);
            previewCamera.transform.position = previewCameraPosition;

           bool isRightSide = mousePosition.x > Screen.width / 2;
            prevAndDelete_Left.SetActive(!isRightSide);
            prevAndDelete_Right.SetActive(isRightSide);
        }

        Vector3Int tilePosition = previewTilemap.WorldToCell(mouseWorldPosition);

        bool differentTile = tilePosition != currentTilePosition;

        // Update tile positions, show the preview and draw if necessary
        if (differentTile || placed == false && isDrawing && placeMode == PlaceMode.Free && !differentTile)
        {
            previousTilePosition = currentTilePosition;
            currentTilePosition = tilePosition;
            if (differentTile)
                placed = false;

            if (isDrawing && placeMode == PlaceMode.Free)
            {
                placed = true;
                if (!ToolAction(baseMap))
                    DrawItem(baseMap);
            }

            UpdatePreviewTilemap();
        }
    }

    // Updates the preview of the action
    private void UpdatePreviewTilemap()
    {
        if (selectedObj == null)
        {
            // Clear preview tiles if no object selected
            previewTilemap.ClearAllTiles();
            return;
        }

        // Render preview based on current place mode
        switch (placeMode)
        {
            default:
                RendererSingle();               
                break;
            case PlaceMode.Line:
                RendererLine();
                break;
            case PlaceMode.Area:
                RendererArea();
                break;
        }

    }

    // Render single preview tile
    private void RendererSingle()
    {
        previewTilemap.ClearAllTiles();
        previewTilemap.SetTile(currentTilePosition, tileBase);
    }

    // Render line preview based on drawing direction
    private void RendererLine()
    {
        if (!isDrawing)
        {
            previewTilemap.ClearAllTiles();
            previewTilemap.SetTile(currentTilePosition, tileBase);
            return;
        }
        previewTilemap.ClearAllTiles();
        previewTilemap.SetTile(currentTilePosition, tileBase);
        float diffX = Mathf.Abs(holdStartTilePosition.x - currentTilePosition.x);
        float diffY = Mathf.Abs(holdStartTilePosition.y - currentTilePosition.y);
        // [REF-1]
        bool isHorizontal = diffX > diffY;

        if (isHorizontal)
        {
            int x = holdStartTilePosition.x;
            int y = holdStartTilePosition.y;

            while (x != currentTilePosition.x)
            {
                Vector3Int pos = new Vector3Int(x, y, 0);
                previewTilemap.SetTile(pos, tileBase);
                x += (int)Mathf.Sign(currentTilePosition.x - holdStartTilePosition.x);
            }
        }
        else
        {
            int x = holdStartTilePosition.x;
            int y = holdStartTilePosition.y;

            while (y != currentTilePosition.y)
            {
                Vector3Int pos = new Vector3Int(x, y, 0);
                previewTilemap.SetTile(pos, tileBase);
                y += (int)Mathf.Sign(currentTilePosition.y - holdStartTilePosition.y);
            }
        }
    }

    // Render area preview based on drawn rectangle
    private void RendererArea()
    {
        if (!isDrawing)
        {
            previewTilemap.ClearAllTiles();
            previewTilemap.SetTile(currentTilePosition, tileBase);
            return;
        }

        previewTilemap.ClearAllTiles();
        previewTilemap.SetTile(currentTilePosition, tileBase);
        Vector3Int min = Vector3Int.Min(holdStartTilePosition, currentTilePosition);
        Vector3Int max = Vector3Int.Max(holdStartTilePosition, currentTilePosition);

        for (int x = min.x; x <= max.x; x++)
        {
            for (int y = min.y; y <= max.y; y++)
            {
                Vector3Int pos = new Vector3Int(x, y, 0);
                previewTilemap.SetTile(pos, tileBase);
            }
        }
    }
}
