using UnityEngine;
using UnityEngine.Tilemaps;
using System.Collections.Generic;

public class BuildingHistory : MonoBehaviour
{
    // TODO improve history logic
    // List to store the history of building actions
    private List<BuildingHistoryStep> history = new List<BuildingHistoryStep>();
    // Current index in the history list
    private int currIndex = -1;

    // Add a new item to the history, removing any redo steps
    public void AddItem(BuildingHistoryStep item)
    {
        // Remove all steps ahead of the current index
        history.RemoveRange(currIndex + 1, history.Count - (currIndex + 1));
        // Add the new item to the history
        history.Add(item);
        // Update the current index
        currIndex++;
    }

    // Undo the last action
    public void UndoStep()
    {
        // Check if there is a step to undo
        if (currIndex >= 0)
        {
            // Perform the undo action
            history[currIndex].Undo();
            // Move back in the history list
            currIndex--;
        }
    }

    // Redo the last undone action
    public void RedoStep()
    {
        // Check if there is a step to redo
        if (currIndex < history.Count - 1)
        {
            // Move forward in the history list
            currIndex++;
            // Perform the redo action
            history[currIndex].Redo();
        }
    }
}

// Class representing a step in the building history
public class BuildingHistoryStep
{
    // Array of items in this history step
    private BuildingHistoryItem[] historyItems;
    
    // Constructor for multiple tilemaps
    public BuildingHistoryStep(Tilemap[] maps, Vector3Int[] positions, TileBase[] prevTiles, TileBase newTile)
    {
        // Initialize the history items array
        historyItems = new BuildingHistoryItem[positions.Length];
        // Create a history item for each map and position
        for (int i = 0; i < maps.Length; i++)
        {
            historyItems[i] = new BuildingHistoryItem(maps[i], positions[i], prevTiles[i], newTile);
        }
    }

    // Overload constructor for a single Tilemap
    public BuildingHistoryStep(Tilemap maps, Vector3Int[] positions, TileBase[] prevTiles, TileBase newTile)
    {
        // Initialize the history items array
        historyItems = new BuildingHistoryItem[positions.Length];
        // Create a history item for each position
        for (int i = 0; i < positions.Length; i++)
        {
            historyItems[i] = new BuildingHistoryItem(maps, positions[i], prevTiles[i], newTile);
        }
    }

    // Constructor for a single position
    public BuildingHistoryStep(Tilemap map, Vector3Int position, TileBase prevTile, TileBase newTile)
    {
        // Initialize the history items array with one item
        historyItems = new BuildingHistoryItem[1];
        // Create the single history item
        historyItems[0] = new BuildingHistoryItem(map, position, prevTile, newTile);
    }

    // Undo all actions in this history step
    public void Undo()
    {
        foreach (BuildingHistoryItem item in historyItems)
        {
            item.Undo();
        }
    }

    // Redo all actions in this history step
    public void Redo()
    {
        foreach (BuildingHistoryItem item in historyItems)
        {
            item.Redo();
        }
    }
}

// Class representing a single item in a history step
public class BuildingHistoryItem
{
    // The Tilemap the action was performed on
    private Tilemap map;
    // The position of the tile
    private Vector3Int position;
    // The previous tile before the action
    private TileBase prevTile;
    // The new tile after the action
    private TileBase newTile;

    // Constructor to initialize the history item
    public BuildingHistoryItem(Tilemap map, Vector3Int position, TileBase prevTile, TileBase newTile)
    {
        this.map = map;
        this.position = position;
        this.prevTile = prevTile;
        this.newTile = newTile;
    }

    // Undo the action by setting the tile back to the previous tile
    public void Undo()
    {
        map.SetTile(position, prevTile);
    }

    // Redo the action by setting the tile back to the new tile
    public void Redo()
    {
        map.SetTile(position, newTile);
    }
}
