using UnityEngine;
using UnityEngine.Tilemaps;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System;

// Enum to categorize different types of building objects
public enum Category
{
    Terrain,
    Floor,
    Floor_Items, // Spawn point, meta, text etc
    Tool
}

// Enum to specify tool types for building objects
public enum ToolType
{
    None,
    Eraser,
    Fill,
    Start,
    End
}

// ScriptableObject class to define properties and behaviors for building objects
[CreateAssetMenu(fileName = "NewBuildingObject", menuName = "Custom/Building Object")]
public class BuildingObjectBase : ScriptableObject
{
    // Category of the building object
    [SerializeField] private Category category;
    // TileBase associated with the building object
    [SerializeField] private TileBase tileBase;
    // Tool type of the building object
    [SerializeField] private ToolType toolType;
    // Array of tiles for rotation (if applicable)
    [Serializable]
    public struct TileInfo
    {
        public TileBase tile;
        public Sprite sprite;

        public TileInfo(TileBase tile, Sprite sprite)
        {
            this.tile = tile;
            this.sprite = sprite;
        }
    }
    [SerializeField] private List<TileInfo> rotationTiles;

    // Current index for the rotation tile
    [SerializeField] private int index = 0;
    // Item index for the building object
    [SerializeField] public int itemIndex;
    // Flag to indicate if the object has rotation tiles
    [SerializeField] public bool hasRotation = false;

    // Property to get the current tile base
    public TileBase TileBase => GetTile();

    // Property to get the category
    public Category Category => category;

    // Property to get the tool type
    public ToolType ToolType => toolType;

    // Property to check if the object has rotation tiles
    public bool HasRotation => hasRotation;

    // Method to get the current tile based on rotation index
    public TileBase GetTile()
    {
        return hasRotation? rotationTiles[index].tile : tileBase;
    }

    // Method to update the rotation index
    public void UpdateIndex()
    {
        Debug.Log("UpdateIndex");
        if (hasRotation)
        {
            index = (index + 1) % rotationTiles.Count;
        }
    }

    // Method to get the sprite of the current tile
    public Sprite GetTileSprite()
    {
        if (hasRotation)
        {
            return rotationTiles[index].sprite;
        }

        return null;
    }
}
