using UnityEngine;
using UnityEngine.UI;
using TMPro;
using System.Collections;

public class ItemToolTip : MonoBehaviour
{
    [Header("UI References")]
    [SerializeField] private TextMeshProUGUI titleText;
    [SerializeField] private TextMeshProUGUI descriptionText;
    [SerializeField] private GameObject extraData;
    [SerializeField] private GameObject extendedView;
    [SerializeField] private Animator animator;
    [SerializeField] private GameObject targetObject;

    [Header("Settings")]
    [SerializeField] private float autoCloseDelay = 1f;

    private Coroutine closeCoroutine;
    private bool isExtended = false;

    // Close the tooltip
    public void CloseToolTip()
    {
        // Hide tooltip
        extendedView.SetActive(false);
        extraData.SetActive(true);
        isExtended = false;

        // Only start the coroutine if it's not already running
        if (closeCoroutine != null)
        {
            StopCoroutine(closeCoroutine);
        }
        closeCoroutine = StartCoroutine(AutoCloseAfterDelay());
    }

    // Update the tooltip with new data
    public void UpdateToolTip(string title, string description)
    {
        targetObject.SetActive(true);
        animator.SetTrigger("Show");
        titleText.text = title;
        descriptionText.text = description;

        // Show tooltip
        extraData.SetActive(true);
        extendedView.SetActive(false);
        isExtended = false;

        // Cancel any existing coroutine and start a new one
        if (closeCoroutine != null)
        {
            StopCoroutine(closeCoroutine);
        }
        closeCoroutine = StartCoroutine(AutoCloseAfterDelay());
    }

    // Automatically close the tooltip after a delay
    private IEnumerator AutoCloseAfterDelay()
    {
        yield return new WaitForSeconds(autoCloseDelay);
        animator.SetTrigger("Close");

        // Wait for the close animation to finish
        yield return new WaitForSeconds(1f);
        targetObject.SetActive(false);
        extendedView.SetActive(false);
        closeCoroutine = null;
    }

    // Toggle between extended and default view
    public void ToggleExtendedView()
    {
        isExtended = !isExtended;
        if (isExtended)
        {
            animator.SetTrigger("Show");
            // Switch to extended view
            extendedView.SetActive(true);
            Debug.Log(extendedView.activeSelf);
            extraData.SetActive(false);
            Debug.Log("Extended view activated");

            // Cancel auto-close
            if (closeCoroutine != null)
            {
                StopCoroutine(closeCoroutine);
            }
        }
        else
        {
            // Revert to default view
            extendedView.SetActive(false);

            Debug.Log(extendedView.activeSelf);
            extraData.SetActive(true);

            // Restart auto-close
            closeCoroutine = StartCoroutine(AutoCloseAfterDelay());
        }
    }
}
