using UnityEngine;
using System.Collections.Generic;

public class CourseScore : MonoBehaviour
{
    // Dictionary to store hole number and its corresponding score (strokes)
    private Dictionary<int, int> holeScores = new Dictionary<int, int>();

    // Base value for calculating the score
    private float baseValue = 10000f;

    private Dictionary<string, Score> levelsScores = null;

    public void Awake()
    {
        if (ES3.KeyExists(LocalSaveKeys.LOCAL_MAPS_KEY, LocalSaveKeys.LEVELS_SCORE_PATH))
        {
            levelsScores = ES3.Load<Dictionary<string, Score>>(
                LocalSaveKeys.LOCAL_MAPS_KEY, 
                LocalSaveKeys.LEVELS_SCORE_PATH
            );
        }
        else
        {
            Debug.LogWarning("Save file or key does not exist. Initializing an empty dictionary.");
            levelsScores = new Dictionary<string, Score>();
        }
    }

    public struct Score
    {
        public int totalStrokes;
        public int totalScore;
        public float totalTime;

        public Score(int totalStrokes, int totalScore, float totalTime)
        {
            this.totalStrokes = totalStrokes;
            this.totalScore = totalScore;
            this.totalTime = totalTime;
        }
    }

    public string GetScoreText(int strokes, int par)
    {
        Debug.Log("Strokes: " + strokes + " Par: " + par);
        if (strokes == par - 4)
        {
            return "Condor";
        }
        else if (strokes == par - 3)
        {
            return "Albatross";
        }
        else if (strokes == par - 2)
        {
            return "Eagle";
        }
        else if (strokes == par - 1)
        {
            return "Birdie";
        }
        else if (strokes == par)
        {
            return "Par";
        }
        else if (strokes == par + 1)
        {
            return "Bogey";
        }
        else if (strokes == par + 2)
        {
            return "Double Bogey";
        }
        else if (strokes == par + 3)
        {
            return "Triple Bogey";
        }
        else if (strokes > par + 3)
        {
            return "Over Par";
        }
        else
        {
            return "Under Par";
        }
    }

    // Update the score for the given hole
    public void SetHoleScore(int holeNum, int strokes)
    {
      holeScores[holeNum] = strokes;
    }

    // Get the dictionary of hole scores
    public Dictionary<int, int> GetHoleScores()
    {
      return holeScores;
    }

    public int GetTotalStroke()
    {
      int totalStrokes = 0;
      foreach (var hole in holeScores)
      {
        totalStrokes += hole.Value;
      }
      return totalStrokes;
    }

    // Calculate the total score
    public int GetTotalScore(int totalStrokes, float totalTime)
    {
        if (totalStrokes <= 0 || totalTime <= 0)
        {
            Debug.LogWarning("Strikes and time must be greater than zero.");
            return 0;
        }

        // Calculate the score
        float score = baseValue / (totalStrokes * totalTime);
        return Mathf.RoundToInt(Mathf.Max(score, 0)); // Ensure score is not negative
    }

    // Save the score
    public void SaveScore(string _levelName, float _totalTime)
    {
        Score score = new Score
        {
            totalStrokes = GetTotalStroke(),
            totalTime = _totalTime,
            totalScore = GetTotalScore(GetTotalStroke(), _totalTime),
        };

        // Check if the level already exists in the dictionary
        if (levelsScores.ContainsKey(_levelName))
        {
            // Retrieve the existing score for the level
            var existingScore = levelsScores[_levelName];

            // Update score properties only if the new score is better
            existingScore.totalScore = Mathf.Max(existingScore.totalScore, score.totalScore);
            existingScore.totalTime = Mathf.Min(existingScore.totalTime, score.totalTime);
            existingScore.totalStrokes = Mathf.Min(existingScore.totalStrokes, score.totalStrokes);

            // Save the updated score back to the dictionary
            levelsScores[_levelName] = existingScore;
        }
        else
        {
            levelsScores.Add(_levelName, score);
        }

        // Save the data using EasySave3
        ES3.Save(LocalSaveKeys.LOCAL_MAPS_KEY, levelsScores, LocalSaveKeys.LEVELS_SCORE_PATH);
            
    }
}
